/*
 * Copyright 1998-2013 University Corporation for Atmospheric Research/Unidata
 *
 *  Portions of this software were developed by the Unidata Program at the
 *  University Corporation for Atmospheric Research.
 *
 *  Access and use of this software shall impose the following obligations
 *  and understandings on the user. The user is granted the right, without
 *  any fee or cost, to use, copy, modify, alter, enhance and distribute
 *  this software, and any derivative works thereof, and its supporting
 *  documentation for any purpose whatsoever, provided that this entire
 *  notice appears in all copies of the software, derivative works and
 *  supporting documentation.  Further, UCAR requests that the user credit
 *  UCAR/Unidata in any publications that result from the use of this
 *  software or in any product that includes this software. The names UCAR
 *  and/or Unidata, however, may not be used in any advertising or publicity
 *  to endorse or promote any products or commercial entity unless specific
 *  written permission is obtained from UCAR/Unidata. The user also
 *  understands that UCAR/Unidata is not obligated to provide the user with
 *  any support, consulting, training or assistance of any kind with regard
 *  to the use, operation and performance of this software nor to provide
 *  the user with any updates, revisions, new versions or "bug fixes."
 *
 *  THIS SOFTWARE IS PROVIDED BY UCAR/UNIDATA "AS IS" AND ANY EXPRESS OR
 *  IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 *  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 *  DISCLAIMED. IN NO EVENT SHALL UCAR/UNIDATA BE LIABLE FOR ANY SPECIAL,
 *  INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *  FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *  NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *  WITH THE ACCESS, USE OR PERFORMANCE OF THIS SOFTWARE.
 */

package ucar.nc2.iosp.bufr;

import ucar.nc2.iosp.bufr.tables.*;
import ucar.nc2.wmo.CommonCodeTable;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Formatter;
import java.util.List;

/**
 * Look up info in BUFR tables.
 * Allows local center overrides for BUFR tables
 *
 * @author caron
 * @since 8/22/13
 */
public class BufrTableLookup {

  static public BufrTableLookup factory(Message m) throws IOException {
    return new BufrTableLookup(m.is.getBufrEdition(), m.ids.getCenterId(), m.ids.getSubCenterId(),
            m.ids.getMasterTableId(), m.ids.getMasterTableVersion(), m.ids.getLocalTableVersion(),
            m.ids.getCategory(), m.ids.getSubCategory(), m.ids.getLocalSubCategory());
  }

  /* static public BufrTableLookup factory(int bufrEdition, int center, int subCenter, int masterId, int masterVersion, int localVersion,
                                       int category, int subCategory, int localSubCategory) {
    return new BufrTableLookup(bufrEdition, center, subCenter, masterId, masterVersion, localVersion, category, subCategory, localSubCategory);
  } */

  //////////////////////////////////////////////////////////

  private int center, subCenter, masterId, masterVersion, localVersion, bufrEdition, category, subCategory, localSubCategory;

  private BufrTableLookup(int bufrEdition, int center, int subCenter, int masterId, int masterVersion, int localVersion,
                         int category, int subCategory, int localSubCategory) throws IOException {
    this.bufrEdition = bufrEdition;
    this.center = center;
    this.subCenter = subCenter;
    this.masterId = masterId;
    this.masterVersion = masterVersion;
    this.localVersion = localVersion;
    this.category = category;
    this.subCategory = subCategory;
    this.localSubCategory = localSubCategory;

    tlookup = new TableLookup(center, subCenter, masterVersion, localVersion, category);
  }

  public int getBufrEdition() {
    return bufrEdition;
  }

  public int getCenter() {
    return center;
  }

  public int getSubCenter() {
    return subCenter;
  }

  public int getMasterTableId() {
    return masterId;
  }

  public int getMasterTableVersion() {
    return masterVersion;
  }

  public int getLocalTableVersion() {
    return localVersion;
  }

  public int getCategory() {
    return category;
  }

  public int getSubCategory() {
    return subCategory;
  }

  public int getLocalSubCategory() {
    return localSubCategory;
  }

  public String getCenterName() {
    String name = CommonCodeTable.getCenterNameBufr(getCenter(), getBufrEdition());
    String subname = CommonCodeTable.getSubCenterName(getCenter(), getSubCenter());
    if (subname != null) name = name +" / " + subname;
    return getCenter() + "." + getSubCenter() + " (" + name + ")";
  }

  public String getCenterNo() {
    return getCenter() + "." + getSubCenter();
  }

  public String getTableName() {
    return getMasterTableId() + "." + getMasterTableVersion() + "." + getLocalTableVersion();
  }

  public String getCategoryFullName() { // throws IOException {
    String catName = getCategoryName();
    String subcatName = getSubCategoryName();

    if (subcatName != null)
      return getCategoryNo() + "="+ catName + " / " + subcatName;
    else
      return getCategoryNo() + "="+ catName;
  }

  public String getSubCategoryName() { // throws IOException {
    String subcatName = null;
    if (center == 7)
      subcatName = NcepTable.getDataSubcategory(getCategory(), getSubCategory());
    if (subcatName == null)
      subcatName = CommonCodeTable.getDataSubcategoy(getCategory(), getSubCategory());
    return subcatName;
  }


  public String getCategoryName() {
    return TableA.getDataCategory(getCategory());
  }

  public String getCategoryNo() {
    String result = getCategory() + "." + getSubCategory();
    if (getLocalSubCategory() >= 0) result += "." + getLocalSubCategory();
    return result;
  }

  ///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
  private TableLookup tlookup;

  private void init() {
  }

  public void setTableLookup(TableLookup tlookup) {
    this.tlookup = tlookup;
  }

  public TableB.Descriptor getDescriptorTableB(short fxy) {
    return tlookup.getDescriptorTableB(fxy);
  }

  public TableD.Descriptor getDescriptorTableD(short fxy) {
    return tlookup.getDescriptorTableD(fxy);
  }

  public String getWmoTableBName() {
    return tlookup.getWmoTableBName();
  }

  public String getLocalTableBName() {
    return tlookup.getLocalTableBName();
  }

  public String getLocalTableDName() {
    return tlookup.getLocalTableDName();
  }

  public String getWmoTableDName() {
    return tlookup.getWmoTableDName();
  }

  public BufrTables.Mode getMode() {
    return tlookup.getMode();
  }

  public void showMissingFields(List<Short> ddsList, Formatter out) throws IOException {
    for (short fxy : ddsList) {
      int f = (fxy & 0xC000) >> 14;
      if (f == 3) {
        List<Short> sublist = getDescriptorListTableD(fxy);
        if (sublist == null) out.format("%s, ", Descriptor.makeString(fxy));
        else showMissingFields(sublist, out);

      } else if (f == 0) {  // skip the 2- operators for now
        TableB.Descriptor b = getDescriptorTableB(fxy);
        if (b == null) out.format("%s, ", Descriptor.makeString(fxy));
      }
    }
  }

  public List<String> getDescriptorListTableD(String fxy) {
    short id = Descriptor.getFxy(fxy);
    List<Short> seq = getDescriptorListTableD(id);
    if (seq == null) return null;
    List<String> result = new ArrayList<String>(seq.size());
    for (Short s : seq)
      result.add(Descriptor.makeString(s));
    return result;
  }

  public List<Short> getDescriptorListTableD(short id) {
    TableD.Descriptor d = getDescriptorTableD(id);
    if (d != null)
      return d.getSequence();
    return null;
  }
}
