/*
 * Decompiled with CFR 0.152.
 */
package ucar.unidata.io;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.WritableByteChannel;
import java.time.Duration;
import java.util.concurrent.ExecutionException;
import javax.annotation.Nonnull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import ucar.unidata.io.RandomAccessFile;
import ucar.unidata.io.ReadableRemoteFile;

public abstract class RemoteRandomAccessFile
extends RandomAccessFile
implements ReadableRemoteFile {
    private static final Logger logger = LoggerFactory.getLogger(RemoteRandomAccessFile.class);
    protected static final long defaultMaxReadCacheSize = 0xA00000L;
    protected static final int defaultRemoteFileBufferSize = 262144;
    protected static final int defaultRemoteFileTimeout = 10000;
    private static final long defaultReadCacheTimeToLive = 30000L;
    protected final String url;
    private final boolean readCacheEnabled;
    private final int readCacheBlockSize;
    private final LoadingCache<Long, byte[]> readCache;

    protected RemoteRandomAccessFile(String url, int bufferSize, long maxRemoteCacheSize) {
        super(bufferSize);
        this.url = url;
        this.file = null;
        this.location = url;
        int minimumCacheActivationSize = 2 * bufferSize;
        if (maxRemoteCacheSize >= (long)minimumCacheActivationSize) {
            this.readCacheBlockSize = bufferSize;
            long numberOfCacheBlocks = maxRemoteCacheSize / (long)this.readCacheBlockSize + 1L;
            this.readCache = this.initCache(numberOfCacheBlocks, Duration.ofMillis(30000L));
            this.readCacheEnabled = true;
        } else {
            this.readCacheBlockSize = -1;
            this.readCacheEnabled = false;
            this.readCache = null;
        }
    }

    private LoadingCache<Long, byte[]> initCache(long maximumNumberOfCacheBlocks, Duration timeToLive) {
        CacheBuilder cb = CacheBuilder.newBuilder().maximumSize(maximumNumberOfCacheBlocks).expireAfterWrite(timeToLive);
        if (debugAccess) {
            cb.recordStats();
        }
        return cb.build((CacheLoader)new CacheLoader<Long, byte[]>(){

            public byte[] load(@Nonnull Long key) throws IOException {
                return RemoteRandomAccessFile.this.readRemoteCacheSizedChunk(key);
            }
        });
    }

    @Override
    protected int read_(long pos, byte[] buff, int offset, int len) throws IOException {
        return this.readCacheEnabled ? this.readFromCache(pos, buff, offset, len) : this.readRemote(pos, buff, offset, len);
    }

    private int readFromCache(long pos, byte[] buff, int offset, int len) throws IOException {
        long firstCacheBlockNumber = pos / (long)this.readCacheBlockSize;
        long lastCacheBlockNumber = (pos + (long)len) / (long)this.readCacheBlockSize;
        int totalBytesRead = 0;
        int currentOffsetIntoBuffer = offset;
        totalBytesRead += this.readCacheBlockPartial(pos, buff, currentOffsetIntoBuffer, true);
        currentOffsetIntoBuffer += totalBytesRead;
        if (totalBytesRead < len && firstCacheBlockNumber != lastCacheBlockNumber) {
            long currentCacheBlockNumber;
            for (currentCacheBlockNumber = firstCacheBlockNumber + 1L; currentCacheBlockNumber < lastCacheBlockNumber; ++currentCacheBlockNumber) {
                totalBytesRead += this.readCacheBlockFull(currentCacheBlockNumber, currentOffsetIntoBuffer, buff);
                currentOffsetIntoBuffer += this.readCacheBlockSize;
            }
            logger.debug("Number of full cache block reads: {}", (Object)(currentCacheBlockNumber - firstCacheBlockNumber - 1L));
            if (totalBytesRead < len) {
                totalBytesRead += this.readCacheBlockPartial(pos + (long)totalBytesRead, buff, currentOffsetIntoBuffer, false);
            }
        }
        return totalBytesRead;
    }

    private int readCacheBlockPartial(long pos, byte[] buff, int positionInBuffer, boolean fillForward) throws IOException {
        int sizeToCopy;
        int offsetIntoCacheBlock;
        byte[] src;
        long cacheBlockNumber = pos / (long)this.readCacheBlockSize;
        try {
            src = (byte[])this.readCache.get((Object)cacheBlockNumber);
        }
        catch (ExecutionException ee) {
            throw new IOException("Error obtaining data from the remote data read cache.", ee);
        }
        long posCacheBlockStart = cacheBlockNumber * (long)this.readCacheBlockSize;
        if (fillForward) {
            offsetIntoCacheBlock = Math.toIntExact(pos - posCacheBlockStart);
            sizeToCopy = this.readCacheBlockSize - offsetIntoCacheBlock;
        } else {
            offsetIntoCacheBlock = 0;
            sizeToCopy = Math.toIntExact(pos - posCacheBlockStart);
        }
        long toEof = this.length() - pos;
        sizeToCopy = Math.toIntExact(Math.min((long)sizeToCopy, toEof));
        sizeToCopy = Math.toIntExact(Math.min(sizeToCopy, buff.length - positionInBuffer));
        logger.debug("Requested {} bytes from the cache block (cache block size upper limit: {} bytes.)", (Object)sizeToCopy, (Object)this.readCacheBlockSize);
        logger.debug("Actual size of the cache block: {} bytes.", (Object)src.length);
        logger.debug("Offset into cache block to begin copy: {} bytes.", (Object)offsetIntoCacheBlock);
        logger.debug("Total size of the destination buffer: {} bytes.", (Object)buff.length);
        logger.debug("Position in buffer to place the copy from the cache: {} bytes.", (Object)positionInBuffer);
        logger.debug("Trying to fit {} bytes from the cache into {} bytes of the destination.", (Object)(src.length - offsetIntoCacheBlock), (Object)(buff.length - positionInBuffer));
        System.arraycopy(src, offsetIntoCacheBlock, buff, positionInBuffer, sizeToCopy);
        return sizeToCopy;
    }

    private int readCacheBlockFull(long cacheBlockNumber, int positionInBuffer, byte[] buff) throws IOException {
        byte[] src;
        try {
            src = (byte[])this.readCache.get((Object)cacheBlockNumber);
        }
        catch (ExecutionException ee) {
            throw new IOException("Error obtaining data from the remote data read cache.", ee);
        }
        System.arraycopy(src, 0, buff, positionInBuffer, this.readCacheBlockSize);
        return this.readCacheBlockSize;
    }

    private byte[] readRemoteCacheSizedChunk(Long cacheBlockNumber) throws IOException {
        long position = cacheBlockNumber * (long)this.readCacheBlockSize;
        long toEOF = this.length() - position;
        long bytesToRead = toEOF < (long)this.readCacheBlockSize ? toEOF : (long)this.readCacheBlockSize;
        int bytes = Math.toIntExact(bytesToRead);
        byte[] buffer = new byte[bytes];
        this.readRemote(position, buffer, 0, bytes);
        return buffer;
    }

    @Override
    public long readToByteChannel(WritableByteChannel dest, long offset, long nbytes) throws IOException {
        int n = (int)nbytes;
        byte[] buff = new byte[n];
        int done = this.read_(offset, buff, 0, n);
        dest.write(ByteBuffer.wrap(buff));
        return done;
    }

    @Override
    public void close() throws IOException {
        this.closeRemote();
        super.close();
        if (this.readCache != null) {
            this.readCache.invalidateAll();
            if (debugAccess) {
                logger.info(this.readCache.stats().toString());
            }
        }
    }
}

