/*
 * Copyright (c) 1998-2018 University Corporation for Atmospheric Research/Unidata
 * See LICENSE for license information.
 */
package ucar.ma2;

import ucar.nc2.util.Indent;
import java.util.Formatter;
import java.util.List;

/**
 * A container for a Structure's data.
 * Is normally contained within an ArrayStructure, which is an Array of StructureData.
 * This is the abstract supertype for all implementations.
 *
 * <pre>
   for (Iterator iter = sdata.getMembers().iterator(); iter.hasNext(); ) {
      StructureMembers.Member m = (StructureMembers.Member) iter.next();
      Array sdataArray = sdata.getArray(m);
      ...
   }
 * </pre>
 * 
 * General ways to access data in an StructureData are:
 * 
 * <pre>
 *  Array getArray(Member m)
 * </pre>
 * 
 * <pre>
 *  Array getArray(String memberName)
 * </pre>
 * 
 * The following will return an object of type Byte, Char, Double, Float, Int, Long, Short, String, or Structure,
 * depending
 * upon the member type:
 * 
 * <pre>
 *  Object getScalarObject( Member m)
 * </pre>
 * 
 * A number of convenience routines may be able to avoid extra Object creation, and so are recommended for efficiency.
 * These require that you know the data types of the member data, but they are the most efficent:
 * 
 * <pre>
  getScalarXXX(int recnum, Member m)
  getJavaArrayXXX(int recnum, Member m)
 * </pre>
 * 
 * where XXX is Byte, Char, Double, Float, Int, Long, Short, or String. For members that are themselves Structures,
 * the equivilent is:
 * 
 * <pre>
  StructureData getScalarStructure(int recnum, Member m)
  ArrayStructure getArrayStructure(int recnum, Member m)
 * </pre>
 * 
 * These will return any compatible type as a double or float, but may have extra overhead when the types dont match:
 * 
 * <pre>
  convertScalarXXX(int recnum, Member m)
  convertJavaArrayXXX(int recnum, Member m)
 * </pre>
 * 
 * where XXX is Double or Float
 *
 * @deprecated use ucar.array.StructureData.
 */
@Deprecated
public abstract class StructureData {
  public static final StructureData EMPTY = new StructureDataW(StructureMembers.builder().setName("empty").build());

  protected StructureMembers members;

  protected StructureData(StructureMembers members) {
    this.members = members;
  }

  /**
   * @return name of Structure
   */
  public String getName() {
    return members.getName();
  }

  /**
   * @return StructureMembers object
   */
  public StructureMembers getStructureMembers() {
    return members;
  }

  /**
   * @return List of StructureMembers.Member
   */
  public List<StructureMembers.Member> getMembers() {
    return members.getMembers();
  }

  /**
   * Find a member by its name.
   *
   * @param memberName find member with this name
   * @return StructureMembers.Member matching the name, or null if not found
   */
  public StructureMembers.Member findMember(String memberName) {
    return members.findMember(memberName);
  }


  //////////////////////////////////////////////////////////////////////////////////////////////////////


  /**
   * Get member data array of any type as an Array.
   * 
   * @param m get data from this StructureMembers.Member.
   * @return Array values.
   */
  public abstract Array getArray(StructureMembers.Member m);

  /**
   * Get member data array of any type as an Array.
   * For more efficiency, use getScalarXXX(Member) or getJavaArrayXXX(Member) is possible.
   * 
   * @param memberName name of member Variable.
   * @return member data array of any type as an Array.
   * @throws IllegalArgumentException if name is not legal member name.
   */
  public Array getArray(String memberName) {
    StructureMembers.Member m = members.findMember(memberName);
    if (m == null)
      throw new IllegalArgumentException("illegal member name =" + memberName);
    return getArray(m);
  }

  /**
   * Get member data array of any type as an Object, eg, Float, Double, String etc.
   * 
   * @param memberName name of member Variable.
   * @return value as Float, Double, etc..
   */
  public Object getScalarObject(String memberName) {
    StructureMembers.Member m = members.findMember(memberName);
    if (m == null)
      throw new IllegalArgumentException("illegal member name =" + memberName);
    return getScalarObject(m);
  }

  /**
   * Get member data array of any type as an Object, eg, Float, Double, String etc.
   * 
   * @param m get data from this StructureMembers.Member.
   * @return value as Float, Double, etc..
   */
  public Object getScalarObject(StructureMembers.Member m) {
    DataType dataType = m.getDataType();
    // boolean isScalar = m.isScalar();

    if (dataType == DataType.DOUBLE) {
      return getScalarDouble(m);

    } else if (dataType == DataType.FLOAT) {
      return getScalarFloat(m);

    } else if (dataType.getPrimitiveClassType() == byte.class) {
      return getScalarByte(m);

    } else if (dataType.getPrimitiveClassType() == short.class) {
      return getScalarShort(m);

    } else if (dataType.getPrimitiveClassType() == int.class) {
      return getScalarInt(m);

    } else if (dataType.getPrimitiveClassType() == long.class) {
      return getScalarLong(m);

    } else if (dataType == DataType.CHAR) {
      return getScalarString(m);

    } else if (dataType == DataType.STRING) {
      return getScalarString(m);

    } else if (dataType == DataType.STRUCTURE) {
      return getScalarStructure(m);

    } else if (dataType == DataType.SEQUENCE) {
      return getArraySequence(m);
    }

    throw new RuntimeException("Dont have implemenation for " + dataType);
  }

  /////////////////////////////////////////////////////////////////////////////////////////////


  /**
   * Get scalar value as a float, with conversion as needed. Underlying type must be convertible to float.
   * 
   * @param memberName name of member Variable. Must be convertible to float.
   * @return scalar value as a float
   * @throws ForbiddenConversionException if not convertible to float.
   */
  public float convertScalarFloat(String memberName) {
    StructureMembers.Member m = members.findMember(memberName);
    if (m == null)
      throw new IllegalArgumentException("illegal member name =" + memberName);
    return convertScalarFloat(m);
  }

  /**
   * Get scalar value as a float, with conversion as needed. Underlying type must be convertible to float.
   * 
   * @param m member Variable.
   * @return scalar value as a float
   * @throws ForbiddenConversionException if not convertible to float.
   */
  public abstract float convertScalarFloat(StructureMembers.Member m);

  /**
   * Get scalar value as a double, with conversion as needed. Underlying type must be convertible to double.
   * 
   * @param memberName name of member Variable. Must be convertible to double.
   * @return scalar value as a double
   * @throws ForbiddenConversionException if not convertible to double.
   */
  public double convertScalarDouble(String memberName) {
    StructureMembers.Member m = members.findMember(memberName);
    if (m == null)
      throw new IllegalArgumentException("illegal member name =" + memberName);
    return convertScalarDouble(m);
  }

  /**
   * Get scalar value as a double, with conversion as needed. Underlying type must be convertible to double.
   * 
   * @param m member Variable.
   * @return scalar value as a double
   * @throws ForbiddenConversionException if not convertible to double.
   */
  public abstract double convertScalarDouble(StructureMembers.Member m);

  /**
   * Get scalar value as a int, with conversion as needed. Underlying type must be convertible to int.
   * 
   * @param memberName name of member Variable. Must be convertible to double.
   * @return scalar value as a int
   * @throws ForbiddenConversionException if not convertible to int.
   */
  public int convertScalarInt(String memberName) {
    StructureMembers.Member m = members.findMember(memberName);
    if (m == null)
      throw new IllegalArgumentException("illegal member name =" + memberName);
    return convertScalarInt(m);
  }

  /**
   * Get scalar value as a int, with conversion as needed. Underlying type must be convertible to int.
   * 
   * @param m member Variable.
   * @return scalar value as a int
   * @throws ForbiddenConversionException if not convertible to int.
   */
  public abstract int convertScalarInt(StructureMembers.Member m);

  /**
   * Get scalar value as a int, with conversion as needed. Underlying type must be convertible to int.
   * 
   * @param memberName name of member Variable. Must be convertible to double.
   * @return scalar value as a int
   * @throws ForbiddenConversionException if not convertible to int.
   */
  public long convertScalarLong(String memberName) {
    StructureMembers.Member m = members.findMember(memberName);
    if (m == null)
      throw new IllegalArgumentException("illegal member name =" + memberName);
    return convertScalarLong(m);
  }

  /**
   * Get scalar value as a int, with conversion as needed. Underlying type must be convertible to int.
   * 
   * @param m member Variable.
   * @return scalar value as a int
   * @throws ForbiddenConversionException if not convertible to int.
   */
  public abstract long convertScalarLong(StructureMembers.Member m);

  /////////////////////////////////////////////////////////////////////////////////////////////

  /**
   * Get member data of type double.
   * 
   * @param memberName name of member Variable. Must be of type double.
   * @throws IllegalArgumentException if name is not legal member name.
   * @return scalar value as a double
   */
  public double getScalarDouble(String memberName) {
    StructureMembers.Member m = members.findMember(memberName);
    if (m == null)
      throw new IllegalArgumentException("illegal member name =" + memberName);
    return getScalarDouble(m);
  }

  /**
   * Get member data of type double.
   * 
   * @param m get data from this StructureMembers.Member. Must be of type double.
   * @return scalar double value
   */
  public abstract double getScalarDouble(StructureMembers.Member m);

  /**
   * Get java double array for a member of type double.
   * 
   * @param memberName name of member Variable. Must be of type double.
   * @return 1D java array of doubles
   */
  public double[] getJavaArrayDouble(String memberName) {
    StructureMembers.Member m = members.findMember(memberName);
    if (m == null)
      throw new IllegalArgumentException("illegal member name =" + memberName);
    return getJavaArrayDouble(m);
  }

  /**
   * Get java double array for a member of type double.
   * 
   * @param m get data from this StructureMembers.Member. Must be of type double.
   * @return 1D java array of doubles
   */
  public abstract double[] getJavaArrayDouble(StructureMembers.Member m);

  ////////////////

  /**
   * Get member data of type float.
   * 
   * @param memberName name of member Variable. Must be of type float.
   * @return scalar float value
   * @throws IllegalArgumentException if name is not legal member name.
   */
  public float getScalarFloat(String memberName) {
    StructureMembers.Member m = members.findMember(memberName);
    if (m == null)
      throw new IllegalArgumentException("illegal member name =" + memberName);
    return getScalarFloat(m);
  }

  /**
   * Get member data of type float.
   * 
   * @param m get data from this StructureMembers.Member. Must be of type float.
   * @return scalar double value
   */
  public abstract float getScalarFloat(StructureMembers.Member m);

  /**
   * Get java float array for a member of type float.
   * 
   * @param memberName name of member Variable. Must be of type float.
   * @return 1D java array of floats
   */
  public float[] getJavaArrayFloat(String memberName) {
    StructureMembers.Member m = members.findMember(memberName);
    if (m == null)
      throw new IllegalArgumentException("illegal member name =" + memberName);
    return getJavaArrayFloat(m);
  }

  /**
   * Get java float array for a member of type float.
   * 
   * @param m get data from this StructureMembers.Member. Must be of type float.
   * @return 1D java array of floats
   */
  public abstract float[] getJavaArrayFloat(StructureMembers.Member m);

  /////

  /**
   * Get member data of type byte.
   * 
   * @param memberName name of member Variable. Must be of type byte.
   * @return scalar byte value
   * @throws IllegalArgumentException if name is not legal member name.
   */
  public byte getScalarByte(String memberName) {
    StructureMembers.Member m = members.findMember(memberName);
    if (m == null)
      throw new IllegalArgumentException("illegal member name =" + memberName);
    return getScalarByte(m);
  }

  /**
   * Get member data of type byte.
   * 
   * @param m get data from this StructureMembers.Member. Must be of type byte.
   * @return scalar byte value
   */
  public abstract byte getScalarByte(StructureMembers.Member m);

  /**
   * Get java byte array for a member of type byte.
   * 
   * @param memberName name of member Variable. Must be of type byte.
   * @return 1D java array of bytes
   */
  public byte[] getJavaArrayByte(String memberName) {
    StructureMembers.Member m = members.findMember(memberName);
    if (m == null)
      throw new IllegalArgumentException("illegal member name =" + memberName);
    return getJavaArrayByte(m);
  }

  /**
   * Get java byte array for a member of type byte.
   * 
   * @param m get data from this StructureMembers.Member. Must be of type byte.
   * @return 1D java array of bytes
   */
  public abstract byte[] getJavaArrayByte(StructureMembers.Member m);

  /////
  /**
   * Get member data of type int.
   * 
   * @param memberName name of member Variable. Must be of type int.
   * @return scalar int value
   * @throws IllegalArgumentException if name is not legal member name.
   */
  public int getScalarInt(String memberName) {
    StructureMembers.Member m = members.findMember(memberName);
    if (m == null)
      throw new IllegalArgumentException("illegal member name =" + memberName);
    return getScalarInt(m);
  }

  /**
   * Get member data of type int.
   * 
   * @param m get data from this StructureMembers.Member. Must be of type int.
   * @return scalar int value
   */
  public abstract int getScalarInt(StructureMembers.Member m);

  /**
   * Get java int array for a member of type int.
   * 
   * @param memberName name of member Variable. Must be of type int.
   * @return 1D java array of ints
   */
  public int[] getJavaArrayInt(String memberName) {
    StructureMembers.Member m = members.findMember(memberName);
    if (m == null)
      throw new IllegalArgumentException("illegal member name =" + memberName);
    return getJavaArrayInt(m);
  }

  /**
   * Get java int array for a member of type int.
   * 
   * @param m get data from this StructureMembers.Member. Must be of type int.
   * @return 1D java array of ints
   */
  public abstract int[] getJavaArrayInt(StructureMembers.Member m);

  /////
  /**
   * Get member data of type short.
   * 
   * @param memberName name of member Variable. Must be of type short.
   * @return scalar short value
   * @throws IllegalArgumentException if name is not legal member name.
   */
  public short getScalarShort(String memberName) {
    StructureMembers.Member m = members.findMember(memberName);
    if (m == null)
      throw new IllegalArgumentException("illegal member name =" + memberName);
    return getScalarShort(m);
  }

  /**
   * Get member data of type short.
   * 
   * @param m get data from this StructureMembers.Member. Must be of type short.
   * @return scalar short value
   */
  public abstract short getScalarShort(StructureMembers.Member m);

  /**
   * Get java short array for a member of type short.
   * 
   * @param memberName name of member Variable. Must be of type short.
   * @return 1D java array of shorts
   */
  public short[] getJavaArrayShort(String memberName) {
    StructureMembers.Member m = members.findMember(memberName);
    if (m == null)
      throw new IllegalArgumentException("illegal member name =" + memberName);
    return getJavaArrayShort(m);
  }

  /**
   * Get java short array for a member of type short.
   * 
   * @param m get data from this StructureMembers.Member. Must be of type short.
   * @return 1D java array of shorts
   */
  public abstract short[] getJavaArrayShort(StructureMembers.Member m);

  /////
  /**
   * Get member data of type long.
   * 
   * @param memberName name of member Variable. Must be of type long.
   * @return scalar long value
   * @throws IllegalArgumentException if name is not legal member name.
   */
  public long getScalarLong(String memberName) {
    StructureMembers.Member m = members.findMember(memberName);
    if (m == null)
      throw new IllegalArgumentException("illegal member name =" + memberName);
    return getScalarLong(m);
  }

  /**
   * Get member data of type long.
   * 
   * @param m get data from this StructureMembers.Member. Must be of type long.
   * @return scalar long value
   */
  public abstract long getScalarLong(StructureMembers.Member m);

  /**
   * Get java long array for a member of type long.
   * 
   * @param memberName name of member Variable. Must be of type long.
   * @return 1D java array of longs
   */
  public long[] getJavaArrayLong(String memberName) {
    StructureMembers.Member m = members.findMember(memberName);
    if (m == null)
      throw new IllegalArgumentException("illegal member name =" + memberName);
    return getJavaArrayLong(m);
  }

  /**
   * Get java long array for a member of type long.
   * 
   * @param m get data from this StructureMembers.Member. Must be of type long.
   * @return 1D java array of longs
   */
  public abstract long[] getJavaArrayLong(StructureMembers.Member m);

  /////
  /**
   * Get member data of type char.
   * 
   * @param memberName name of member Variable. Must be of type char.
   * @return scalar char value
   * @throws IllegalArgumentException if name is not legal member name.
   */
  public char getScalarChar(String memberName) {
    StructureMembers.Member m = members.findMember(memberName);
    if (m == null)
      throw new IllegalArgumentException("illegal member name =" + memberName);
    return getScalarChar(m);
  }

  /**
   * Get member data of type char.
   * 
   * @param m get data from this StructureMembers.Member. Must be of type char.
   * @return scalar char value
   */
  public abstract char getScalarChar(StructureMembers.Member m);

  /**
   * Get java char array for a member of type char.
   * 
   * @param memberName name of member Variable. Must be of type char.
   * @return 1D java array of chars
   */
  public char[] getJavaArrayChar(String memberName) {
    StructureMembers.Member m = members.findMember(memberName);
    if (m == null)
      throw new IllegalArgumentException("illegal member name =" + memberName);
    return getJavaArrayChar(m);
  }

  /**
   * Get java char array for a member of type char.
   * 
   * @param m get data from this StructureMembers.Member. Must be of type char.
   * @return 1D java array of chars
   */
  public abstract char[] getJavaArrayChar(StructureMembers.Member m);

  /////

  /**
   * Get String value, from rank 0 String or rank 1 char member array.
   * 
   * @param memberName name of member Variable.
   * @return scalar String value
   * @throws IllegalArgumentException if name is not legal member name.
   */
  public String getScalarString(String memberName) {
    StructureMembers.Member m = findMember(memberName);
    if (null == m)
      throw new IllegalArgumentException("Member not found= " + memberName);
    return getScalarString(m);
  }

  /**
   * Get String value, from rank 0 String or rank 1 char member array.
   * 
   * @param m get data from this StructureMembers.Member. Must be of type char or String.
   * @return scalar String value
   */
  public abstract String getScalarString(StructureMembers.Member m);

  /**
   * Get java String array for a member of type String.
   * 
   * @param memberName name of member Variable. Must be of type char or String.
   * @return 1D java array of String
   */
  public String[] getJavaArrayString(String memberName) {
    StructureMembers.Member m = members.findMember(memberName);
    if (m == null)
      throw new IllegalArgumentException("illegal member name =" + memberName);
    return getJavaArrayString(m);
  }

  /**
   * Get java array of Strings for a member of type char or String.
   * 
   * @param m get data from this StructureMembers.Member. Must be of type char or String.
   * @return 1D java array of String
   */
  public abstract String[] getJavaArrayString(StructureMembers.Member m);

  ////

  /**
   * Get member data of type Structure.
   * 
   * @param memberName name of member Variable.
   * @return scalar StructureData value
   * @throws IllegalArgumentException if name is not legal member name.
   */
  public StructureData getScalarStructure(String memberName) {
    StructureMembers.Member m = findMember(memberName);
    if (null == m)
      throw new IllegalArgumentException("Member not found= " + memberName);
    return getScalarStructure(m);
  }

  /**
   * Get member data of type Structure.
   * 
   * @param m get data from this StructureMembers.Member. Must be of type Structure.
   * @return StructureData
   */
  public abstract StructureData getScalarStructure(StructureMembers.Member m);

  /**
   * Get member data of type Structure.
   * 
   * @param memberName name of member Variable.
   * @return array of StructureData
   * @throws IllegalArgumentException if name is not legal member name.
   */
  public ArrayStructure getArrayStructure(String memberName) {
    StructureMembers.Member m = findMember(memberName);
    if (null == m)
      throw new IllegalArgumentException("Member not found= " + memberName);
    return getArrayStructure(m);
  }

  /**
   * Get ArrayStructure for a member of type Structure.
   * 
   * @param m get data from this StructureMembers.Member. Must be of type Structure.
   * @return ArrayStructure
   */
  public abstract ArrayStructure getArrayStructure(StructureMembers.Member m);

  //////

  /**
   * Get ArraySequence for a member of type Sequence.
   * 
   * @param memberName name of member Variable. Must be of type Sequence.
   * @return ArrayStructure
   */
  public ArraySequence getArraySequence(String memberName) {
    StructureMembers.Member m = members.findMember(memberName);
    if (m == null)
      throw new IllegalArgumentException("illegal member name =" + memberName);
    return getArraySequence(m);
  }

  /**
   * Get ArraySequence for a member of type Sequence.
   * 
   * @param m get data from this StructureMembers.Member. Must be of type Sequence.
   * @return ArrayStructure
   */
  public abstract ArraySequence getArraySequence(StructureMembers.Member m);

  ////////////////////////////////////////////////////////////////////////////////////////////////
  // debugging
  public void showInternal(Formatter f, Indent indent) {
    f.format("%sStructureData %s class=%s hash=0x%x%n", indent, members.getName(), this.getClass().getName(),
        hashCode());
  }

  public void showInternalMembers(Formatter f, Indent indent) {
    f.format("%sStructureData %s class=%s hash=0x%x%n", indent, members.getName(), this.getClass().getName(),
        hashCode());
    indent.incr();
    for (StructureMembers.Member m : getMembers())
      m.showInternal(f, indent);
    indent.decr();
  }

  public String toString() {
    return members.toString();
  }

}
