/*
 * Decompiled with CFR 0.152.
 */
package ucar.array;

import com.google.common.base.Preconditions;
import java.util.HashSet;
import java.util.Iterator;
import javax.annotation.concurrent.Immutable;
import ucar.array.Arrays;
import ucar.array.InvalidRangeException;
import ucar.array.Range;
import ucar.array.Section;

@Immutable
final class IndexFn
implements Iterable<Integer> {
    private final int[] shape;
    private final int[] stride;
    private final int rank;
    private final long length;
    private final int offset;
    private final boolean canonicalOrder;

    public int get(int ... index) {
        if (this.rank == 0 && index.length == 1 && index[0] == 0) {
            return 0;
        }
        Preconditions.checkArgument((this.rank == index.length ? 1 : 0) != 0);
        int value = this.offset;
        for (int ii = 0; ii < this.rank; ++ii) {
            if (index[ii] < 0 || index[ii] >= this.shape[ii]) {
                throw new IllegalArgumentException(String.format("IndexFn.get(%s) not inside of shape '%s'", java.util.Arrays.toString(index), java.util.Arrays.toString(this.shape)));
            }
            if (this.shape[ii] < 0) break;
            value += index[ii] * this.stride[ii];
        }
        return value;
    }

    public int getRank() {
        return this.rank;
    }

    public Section getSection() {
        try {
            return new Section(this.odometer(this.offset), this.shape);
        }
        catch (InvalidRangeException e) {
            throw new IllegalStateException(e);
        }
    }

    public int[] getShape() {
        int[] result = new int[this.rank];
        System.arraycopy(this.shape, 0, result, 0, this.rank);
        return result;
    }

    public int getShape(int index) {
        Preconditions.checkArgument((index >= 0 && index < this.rank ? 1 : 0) != 0);
        return this.shape[index];
    }

    @Override
    public Iterator<Integer> iterator() {
        return new Odometer();
    }

    public Iterator<Integer> iterator(int start, long length) {
        return new Odometer(start, length);
    }

    public long length() {
        return this.length;
    }

    public String toString2() {
        StringBuilder sbuff = new StringBuilder();
        boolean first = true;
        for (int i : this) {
            if (!first) {
                sbuff.append(", ");
            }
            sbuff.append(i);
            first = false;
        }
        return sbuff.toString();
    }

    public String toString() {
        return "IndexFn{shape=" + java.util.Arrays.toString(this.shape) + ", stride=" + java.util.Arrays.toString(this.stride) + ", rank=" + this.rank + ", length=" + this.length + ", offset=" + this.offset + ", canonicalOrder=" + this.canonicalOrder + '}';
    }

    boolean isCanonicalOrder() {
        return this.canonicalOrder;
    }

    boolean isVlen() {
        return this.shape.length > 0 && this.shape[this.shape.length - 1] < 0;
    }

    IndexFn flip(int index) {
        Preconditions.checkArgument((index >= 0 && index < this.rank ? 1 : 0) != 0);
        Preconditions.checkArgument((this.shape[index] >= 0 ? 1 : 0) != 0);
        Builder ib = this.toBuilder();
        ib.offset += this.stride[index] * (this.shape[index] - 1);
        ib.stride[index] = -this.stride[index];
        ib.canonicalOrder = false;
        return ib.build();
    }

    IndexFn permute(int[] dims) {
        Preconditions.checkArgument((dims.length == this.shape.length ? 1 : 0) != 0);
        HashSet<Integer> used = new HashSet<Integer>();
        for (int dim : dims) {
            if (dim < 0 || dim >= this.rank) {
                throw new IllegalArgumentException();
            }
            if (used.contains(dim)) {
                throw new IllegalArgumentException();
            }
            used.add(dim);
        }
        boolean isPermuted = false;
        Builder newIndex = this.toBuilder();
        for (int i = 0; i < dims.length; ++i) {
            newIndex.stride[i] = this.stride[dims[i]];
            newIndex.shape[i] = this.shape[dims[i]];
            if (i == dims[i]) continue;
            isPermuted = true;
        }
        newIndex.canonicalOrder = this.canonicalOrder && !isPermuted;
        return newIndex.build();
    }

    IndexFn reshape(int[] shape) {
        Preconditions.checkArgument((Arrays.computeSize(shape) == this.length() ? 1 : 0) != 0);
        return IndexFn.builder(shape).build();
    }

    IndexFn section(Section section) throws InvalidRangeException {
        Preconditions.checkArgument((section.getRank() == this.rank ? 1 : 0) != 0);
        for (int ii = 0; ii < this.rank; ++ii) {
            Range r = section.getRange(ii);
            if (r == null || r == Range.VLEN) continue;
            if (r.first() < 0 || r.first() >= this.shape[ii]) {
                throw new InvalidRangeException("Bad range starting value at index " + ii + " == " + r.first());
            }
            if (r.last() >= 0 && r.last() < this.shape[ii]) continue;
            throw new InvalidRangeException("Bad range ending value at index " + ii + " == " + r.last());
        }
        Builder newindex = IndexFn.builder(this.rank);
        newindex.offset = this.offset;
        int[] newstride = new int[this.rank];
        for (int ii = 0; ii < this.rank; ++ii) {
            Range r = section.getRange(ii);
            if (r == null) {
                newindex.shape[ii] = this.shape[ii];
                newstride[ii] = this.stride[ii];
                continue;
            }
            newindex.shape[ii] = r.length();
            newstride[ii] = this.stride[ii] * r.stride();
            newindex.offset += this.stride[ii] * r.first();
        }
        newindex.setStride(newstride);
        newindex.canonicalOrder = this.canonicalOrder && Arrays.computeSize(newindex.shape) == this.length;
        return newindex.build();
    }

    IndexFn reduce() {
        IndexFn c = this;
        for (int ii = 0; ii < this.rank; ++ii) {
            if (this.shape[ii] != 1) continue;
            IndexFn newc = c.reduce(ii);
            return newc.reduce();
        }
        return c;
    }

    IndexFn reduce(int dim) {
        if (dim < 0 || dim >= this.rank) {
            throw new IllegalArgumentException("illegal reduce dim " + dim);
        }
        if (this.shape[dim] != 1) {
            throw new IllegalArgumentException("illegal reduce dim " + dim + " : length != 1");
        }
        Builder newindex = IndexFn.builder(this.rank - 1);
        newindex.offset = this.offset;
        int[] newstride = new int[this.rank - 1];
        int count = 0;
        for (int ii = 0; ii < this.rank; ++ii) {
            if (ii == dim) continue;
            newindex.shape[count] = this.shape[ii];
            newstride[count] = this.stride[ii];
            ++count;
        }
        newindex.setStride(newstride);
        newindex.canonicalOrder = this.canonicalOrder;
        return newindex.build();
    }

    IndexFn transpose(int index1, int index2) {
        if (index1 < 0 || index1 >= this.rank) {
            throw new IllegalArgumentException();
        }
        if (index2 < 0 || index2 >= this.rank) {
            throw new IllegalArgumentException();
        }
        if (index1 == index2) {
            return this;
        }
        Builder newIndex = this.toBuilder();
        newIndex.stride[index1] = this.stride[index2];
        newIndex.stride[index2] = this.stride[index1];
        newIndex.shape[index1] = this.shape[index2];
        newIndex.shape[index2] = this.shape[index1];
        newIndex.setCanonicalOrder(false);
        return newIndex.build();
    }

    public static Builder builder(int rank) {
        return new Builder(rank);
    }

    public static Builder builder(int[] shape) {
        return new Builder(shape.length).setShape(shape);
    }

    public Builder toBuilder() {
        return new Builder(this.rank).setShape(this.shape).setStride(this.stride).setOffset(this.offset).setCanonicalOrder(this.canonicalOrder);
    }

    private IndexFn(Builder builder) {
        Preconditions.checkNotNull((Object)builder.shape);
        this.rank = builder.shape.length;
        this.shape = new int[this.rank];
        System.arraycopy(builder.shape, 0, this.shape, 0, this.rank);
        if (builder.stride == null) {
            this.stride = new int[this.rank];
            this.length = this.computeStrides(this.shape);
        } else {
            Preconditions.checkArgument((builder.stride.length == this.rank ? 1 : 0) != 0);
            this.stride = new int[this.rank];
            System.arraycopy(builder.stride, 0, this.stride, 0, this.rank);
            this.length = Arrays.computeSize(this.shape);
        }
        this.offset = builder.offset;
        this.canonicalOrder = builder.canonicalOrder;
    }

    private long computeStrides(int[] shape) {
        long product = 1L;
        for (int ii = shape.length - 1; ii >= 0; --ii) {
            int thisDim = shape[ii];
            if (thisDim < 0) continue;
            this.stride[ii] = (int)product;
            product *= (long)thisDim;
        }
        return product;
    }

    int[] odometer(long element) {
        int[] odometer = new int[this.rank];
        for (int dim = 0; dim < this.rank; ++dim) {
            odometer[dim] = (int)(element / (long)this.stride[dim]);
            element -= (long)(odometer[dim] * this.stride[dim]);
        }
        return odometer;
    }

    private class Odometer
    implements Iterator<Integer> {
        private final long nelems;
        private final int[] current;
        private int count = 0;
        private int nextIndex;

        private Odometer() {
            this.nelems = IndexFn.this.length;
            this.current = new int[IndexFn.this.rank];
            this.nextIndex = IndexFn.this.get(this.current);
        }

        private Odometer(int startElement, long nelems) {
            this.nelems = nelems;
            this.current = IndexFn.this.odometer(startElement);
            this.nextIndex = IndexFn.this.get(this.current);
        }

        @Override
        public boolean hasNext() {
            return (long)this.count++ < this.nelems;
        }

        @Override
        public Integer next() {
            int result = this.nextIndex;
            this.nextIndex = this.incr();
            return result;
        }

        private int incr() {
            int digit = IndexFn.this.rank - 1;
            while (digit >= 0) {
                if (IndexFn.this.shape[digit] < 0) {
                    this.current[digit] = -1;
                    continue;
                }
                int n = digit;
                this.current[n] = this.current[n] + 1;
                if (this.current[digit] < IndexFn.this.shape[digit]) break;
                this.current[digit] = 0;
                --digit;
            }
            return IndexFn.this.get(this.current);
        }
    }

    public static class Builder {
        int[] shape;
        int[] stride;
        int offset = 0;
        boolean canonicalOrder = true;

        Builder(int rank) {
            this.shape = new int[rank];
        }

        Builder setShape(int[] shape) {
            this.shape = new int[shape.length];
            System.arraycopy(shape, 0, this.shape, 0, shape.length);
            return this;
        }

        Builder setStride(int[] stride) {
            this.stride = new int[stride.length];
            System.arraycopy(stride, 0, this.stride, 0, stride.length);
            return this;
        }

        Builder setOffset(int offset) {
            this.offset = offset;
            return this;
        }

        Builder setCanonicalOrder(boolean canonicalOrder) {
            this.canonicalOrder = canonicalOrder;
            return this;
        }

        public IndexFn build() {
            return new IndexFn(this);
        }
    }
}

