/*
 * Decompiled with CFR 0.152.
 */
package ucar.array;

import com.google.common.base.Preconditions;
import java.util.Iterator;
import java.util.Objects;
import javax.annotation.Nullable;
import javax.annotation.concurrent.Immutable;
import ucar.array.InvalidRangeException;
import ucar.array.RangeIterator;

@Immutable
public class Range
implements RangeIterator {
    public static final Range EMPTY = new Range();
    public static final Range SCALAR = new Range("SCALAR", 1);
    public static final Range VLEN = new Range("VLEN", -1);
    private final int length;
    private final int first;
    private final int last;
    private final int stride;
    private final String name;

    public static Range make(String name, int len) {
        try {
            return new Range(name, 0, len - 1, 1);
        }
        catch (InvalidRangeException e) {
            throw new RuntimeException(e);
        }
    }

    public static Range make(int first, int last) {
        try {
            return new Range(first, last);
        }
        catch (InvalidRangeException e) {
            throw new RuntimeException(e);
        }
    }

    private Range() {
        this.length = 0;
        this.first = 0;
        this.last = -1;
        this.stride = 1;
        this.name = "EMPTY";
    }

    private Range(String name, int length) {
        assert (length != 0);
        this.name = name;
        this.first = 0;
        this.last = length - 1;
        this.stride = 1;
        this.length = length;
    }

    public Range(int first, int last) throws InvalidRangeException {
        this(null, first, last, 1);
    }

    public Range(int length) {
        assert (length != 0);
        this.name = null;
        this.first = 0;
        this.last = length - 1;
        this.stride = 1;
        this.length = length;
    }

    public Range(String name, int first, int last) throws InvalidRangeException {
        this(name, first, last, 1);
    }

    public Range(int first, int last, int stride) throws InvalidRangeException {
        this(null, first, last, stride);
    }

    public Range(String name, int first, int last, int stride) throws InvalidRangeException {
        if (first < 0) {
            throw new InvalidRangeException("first (" + first + ") must be >= 0");
        }
        if (last < first) {
            throw new InvalidRangeException("last (" + last + ") must be >= first (" + first + ")");
        }
        if (stride < 1) {
            throw new InvalidRangeException("stride (" + stride + ") must be > 0");
        }
        this.name = name;
        this.first = first;
        this.stride = stride;
        this.length = 1 + (last - first) / stride;
        this.last = first + (this.length - 1) * stride;
        assert (stride != 1 || this.last == last);
    }

    private Range(String name, int first, int last, int stride, int length) throws InvalidRangeException {
        if (first < 0) {
            throw new InvalidRangeException("first (" + first + ") must be >= 0");
        }
        if (last < first) {
            throw new InvalidRangeException("last (" + last + ") must be >= first (" + first + ")");
        }
        if (stride < 1) {
            throw new InvalidRangeException("stride (" + stride + ") must be > 0");
        }
        if (length < (1 + last - first) / stride) {
            throw new InvalidRangeException("length (" + length + ") must be > (1 + last - first) / stride");
        }
        this.name = name;
        this.first = first;
        this.last = last;
        this.stride = stride;
        this.length = length;
    }

    public Range copyWithStride(int stride) throws InvalidRangeException {
        if (stride == this.stride) {
            return this;
        }
        return new Range(this.first(), this.last(), stride);
    }

    @Override
    public Range copyWithName(String name) {
        if (name.equals(this.name())) {
            return this;
        }
        try {
            return new Range(name, this.first, this.last, this.stride, this.length);
        }
        catch (InvalidRangeException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    @Nullable
    public String name() {
        return this.name;
    }

    public int first() {
        return this.first;
    }

    public int last() {
        return this.last;
    }

    @Override
    public int length() {
        return this.length;
    }

    public int stride() {
        return this.stride;
    }

    public boolean contains(int want) {
        if (want < this.first()) {
            return false;
        }
        if (want > this.last()) {
            return false;
        }
        if (this.stride == 1) {
            return true;
        }
        return (want - this.first) % this.stride == 0;
    }

    public Range compose(Range r) throws InvalidRangeException {
        if (this.length() == 0 || r.length() == 0) {
            return EMPTY;
        }
        if (this == VLEN || r == VLEN) {
            return VLEN;
        }
        int sr_stride = this.stride * r.stride;
        int sr_first = this.element(r.first());
        int lastx = this.element(r.last());
        int sr_last = Math.min(this.last(), lastx);
        return new Range(this.name, sr_first, sr_last, sr_stride);
    }

    public Range compact() throws InvalidRangeException {
        if (this.stride == 1) {
            return this;
        }
        int first = this.first() / this.stride;
        int last = first + this.length() - 1;
        return new Range(this.name, first, last, 1);
    }

    public int element(int i) throws InvalidRangeException {
        if (i < 0) {
            throw new InvalidRangeException("element idx (" + i + ") must be >= 0");
        }
        if (i >= this.length) {
            throw new InvalidRangeException("element idx (" + i + ") must be < length");
        }
        return this.first + i * this.stride;
    }

    public int index(int want) throws InvalidRangeException {
        if (want < this.first) {
            throw new InvalidRangeException("elem must be >= first");
        }
        int result = (want - this.first) / this.stride;
        if (result > this.length) {
            throw new InvalidRangeException("elem must be <= first = n * stride");
        }
        return result;
    }

    public Range intersect(Range other) throws InvalidRangeException {
        if (this.length() == 0 || other.length() == 0) {
            return EMPTY;
        }
        if (this == VLEN || other == VLEN) {
            return VLEN;
        }
        Preconditions.checkArgument((other.stride == 1 ? 1 : 0) != 0, (Object)"other stride must be 1");
        int first = Math.max(this.first(), other.first());
        int last = Math.min(this.last(), other.last());
        if (first > last) {
            return EMPTY;
        }
        if (this.first() < other.first()) {
            int incr = (other.first() - this.first()) / this.stride;
            first = this.first() + incr * this.stride;
            if (first < other.first()) {
                first += this.stride;
            }
        }
        return new Range(this.name, first, last, this.stride);
    }

    public boolean intersects(Range other) {
        int last;
        if (this.length() == 0 || other.length() == 0) {
            return false;
        }
        if (this == VLEN || other == VLEN) {
            return true;
        }
        int first = Math.max(this.first(), other.first());
        return first <= (last = Math.min(this.last(), other.last()));
    }

    public int getFirstInInterval(int start) {
        if (start > this.last()) {
            return -1;
        }
        if (start <= this.first) {
            return this.first;
        }
        if (this.stride == 1) {
            return start;
        }
        int offset = start - this.first;
        int i = offset / this.stride;
        i = offset % this.stride == 0 ? i : i + 1;
        return this.first + i * this.stride;
    }

    public String toString() {
        if (this.length == 0) {
            return ":";
        }
        if (this.length < 0) {
            return ":";
        }
        return this.first + ":" + this.last() + (this.stride > 1 ? ":" + this.stride : "");
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        Range integers = (Range)o;
        return this.length == integers.length && this.first == integers.first && this.last == integers.last && this.stride == integers.stride;
    }

    public int hashCode() {
        return Objects.hash(this.length, this.first, this.last, this.stride);
    }

    @Override
    public Iterator<Integer> iterator() {
        return new InternalIterator();
    }

    private int elementNC(int i) {
        return this.first + i * this.stride;
    }

    private class InternalIterator
    implements Iterator<Integer> {
        private int current;

        private InternalIterator() {
        }

        @Override
        public boolean hasNext() {
            return this.current < Range.this.length;
        }

        @Override
        public Integer next() {
            return Range.this.elementNC(this.current++);
        }
    }
}

