/*
 * Decompiled with CFR 0.152.
 */
package ucar.nc2.internal.cache;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Formatter;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import javax.annotation.concurrent.GuardedBy;
import javax.annotation.concurrent.ThreadSafe;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import ucar.nc2.dataset.DatasetUrl;
import ucar.nc2.internal.cache.FileCacheIF;
import ucar.nc2.internal.cache.FileCacheable;
import ucar.nc2.internal.cache.FileFactory;
import ucar.nc2.time.CalendarDate;
import ucar.nc2.time.CalendarDateFormatter;
import ucar.nc2.util.CancelTask;

@ThreadSafe
public class FileCache
implements FileCacheIF {
    protected static final Logger log = LoggerFactory.getLogger(FileCache.class);
    protected static final Logger cacheLog = LoggerFactory.getLogger((String)"cacheLogger");
    static final boolean debug = false;
    static final boolean debugPrint = false;
    static final boolean debugCleanup = false;
    private static Timer timer;
    private static final Object lock;
    protected String name;
    protected final int softLimit;
    protected final int minElements;
    protected final int hardLimit;
    protected final long period;
    private final AtomicBoolean disabled = new AtomicBoolean(false);
    protected final AtomicBoolean hasScheduled = new AtomicBoolean(false);
    protected final ConcurrentHashMap<Object, CacheElement> cache;
    protected final ConcurrentHashMap<FileCacheable, CacheElement.CacheFile> files;
    protected final AtomicInteger cleanups = new AtomicInteger();
    protected final AtomicInteger hits = new AtomicInteger();
    protected final AtomicInteger miss = new AtomicInteger();
    protected ConcurrentHashMap<Object, Tracker> track;
    protected boolean trackAll;

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void shutdown() {
        Object object = lock;
        synchronized (object) {
            if (timer != null) {
                timer.cancel();
                cacheLog.info("FileCache.shutdown called%n");
            }
            timer = null;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void scheduleAtFixedRate(TimerTask task, long delay, long period) {
        Object object = lock;
        synchronized (object) {
            if (timer == null) {
                timer = new Timer("FileCache");
            }
            timer.scheduleAtFixedRate(task, delay, period);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void schedule(TimerTask task, long delay) {
        Object object = lock;
        synchronized (object) {
            if (timer == null) {
                timer = new Timer("FileCache");
            }
            timer.schedule(task, delay);
        }
    }

    public FileCache(int minElementsInMemory, int maxElementsInMemory, int period) {
        this("", minElementsInMemory, maxElementsInMemory, -1, period);
    }

    public FileCache(int minElementsInMemory, int softLimit, int hardLimit, int period) {
        this("", minElementsInMemory, softLimit, hardLimit, period);
    }

    public FileCache(String name, int minElementsInMemory, int softLimit, int hardLimit, int period) {
        boolean wantsCleanup;
        this.name = name;
        this.minElements = minElementsInMemory;
        this.softLimit = softLimit;
        this.hardLimit = hardLimit;
        this.period = 1000L * (long)period;
        this.cache = new ConcurrentHashMap(2 * softLimit, 0.75f, 8);
        this.files = new ConcurrentHashMap(4 * softLimit, 0.75f, 8);
        boolean bl = wantsCleanup = period > 0;
        if (wantsCleanup) {
            FileCache.scheduleAtFixedRate(new CleanupTask(), this.period, this.period);
            if (cacheLog.isDebugEnabled()) {
                cacheLog.debug("FileCache " + name + " cleanup every " + period + " secs");
            }
        }
        if (this.trackAll) {
            this.track = new ConcurrentHashMap(5000);
        }
    }

    @Override
    public void disable() {
        this.disabled.set(true);
        this.clearCache(true);
    }

    @Override
    public void enable() {
        this.disabled.set(false);
    }

    public FileCacheable acquire(FileFactory factory, DatasetUrl durl, CancelTask cancelTask) throws IOException {
        return this.acquire(factory, durl.getTrueurl(), durl, -1, cancelTask, null);
    }

    @Override
    public FileCacheable acquire(FileFactory factory, DatasetUrl durl) throws IOException {
        return this.acquire(factory, durl.getTrueurl(), durl, -1, null, null);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public FileCacheable acquire(FileFactory factory, Object hashKey, DatasetUrl location, int buffer_size, CancelTask cancelTask, Object spiObject) throws IOException {
        CacheElement elem;
        FileCacheable ncfile;
        Tracker prev;
        if (null == hashKey) {
            hashKey = location.getTrueurl();
        }
        if (null == hashKey) {
            throw new IllegalArgumentException();
        }
        Tracker t = null;
        if (this.trackAll && (prev = this.track.putIfAbsent(hashKey, t = new Tracker(hashKey))) != null) {
            t = prev;
        }
        if ((ncfile = this.acquireCacheOnly(hashKey)) != null) {
            this.hits.incrementAndGet();
            if (t != null) {
                ++t.hit;
            }
            return ncfile;
        }
        this.miss.incrementAndGet();
        if (t != null) {
            ++t.miss;
        }
        ncfile = factory.open(location, buffer_size, cancelTask, spiObject);
        if (cacheLog.isDebugEnabled()) {
            cacheLog.debug("FileCache " + this.name + " acquire " + hashKey + " " + ncfile.getLocation());
        }
        if (cancelTask != null && cancelTask.isCancel()) {
            if (ncfile != null) {
                ncfile.close();
            }
            return null;
        }
        if (this.disabled.get()) {
            return ncfile;
        }
        Object object = this.cache;
        synchronized (object) {
            elem = this.cache.get(hashKey);
            if (elem == null) {
                this.cache.put(hashKey, new CacheElement(ncfile, hashKey));
            }
        }
        if (elem != null) {
            object = elem;
            synchronized (object) {
                elem.addFile(ncfile);
            }
        }
        boolean needHard = false;
        boolean needSoft = false;
        AtomicBoolean atomicBoolean = this.hasScheduled;
        synchronized (atomicBoolean) {
            if (!this.hasScheduled.get()) {
                int count = this.files.size();
                if (count > this.hardLimit && this.hardLimit > 0) {
                    needHard = true;
                    this.hasScheduled.getAndSet(true);
                } else if (count > this.softLimit && this.softLimit > 0) {
                    this.hasScheduled.getAndSet(true);
                    needSoft = true;
                }
            }
        }
        if (needHard) {
            this.cleanup(this.hardLimit);
        } else if (needSoft) {
            FileCache.schedule(new CleanupTask(), 100L);
        }
        return ncfile;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private FileCacheable acquireCacheOnly(Object hashKey) {
        if (this.disabled.get()) {
            return null;
        }
        CacheElement wantCacheElem = this.cache.get(hashKey);
        if (wantCacheElem == null) {
            return null;
        }
        CacheElement.CacheFile want = null;
        CacheElement cacheElement = wantCacheElem;
        synchronized (cacheElement) {
            for (CacheElement.CacheFile file : wantCacheElem.list) {
                if (!file.isLocked.compareAndSet(false, true)) continue;
                want = file;
                break;
            }
        }
        if (want == null) {
            return null;
        }
        if (want.ncfile != null) {
            boolean changed;
            long lastModified = want.ncfile.getLastModified();
            boolean bl = changed = lastModified != want.lastModified;
            if (cacheLog.isDebugEnabled() && changed) {
                cacheLog.debug("FileCache " + this.name + ": acquire from cache " + hashKey + " " + want.ncfile.getLocation() + " was changed; discard");
            }
            if (changed) {
                this.remove(want);
            }
        }
        if (want.ncfile != null) {
            try {
                want.ncfile.reacquire();
            }
            catch (IOException ioe) {
                if (cacheLog.isDebugEnabled()) {
                    cacheLog.debug("FileCache " + this.name + " acquire from cache " + hashKey + " " + want.ncfile.getLocation() + " failed: " + ioe.getMessage());
                }
                this.remove(want);
            }
        }
        return want.ncfile;
    }

    private void remove(CacheElement.CacheFile want) {
        want.remove();
        this.files.remove(want.ncfile);
        try {
            want.ncfile.setFileCache(null);
            want.ncfile.close();
        }
        catch (IOException e) {
            log.error("close failed on " + want.ncfile.getLocation(), (Throwable)e);
        }
        want.ncfile = null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void eject(Object hashKey) {
        if (this.disabled.get()) {
            return;
        }
        CacheElement wantCacheElem = this.cache.get(hashKey);
        if (wantCacheElem == null) {
            return;
        }
        CacheElement cacheElement = wantCacheElem;
        synchronized (cacheElement) {
            for (CacheElement.CacheFile want : wantCacheElem.list) {
                this.files.remove(want.ncfile);
                try {
                    want.ncfile.setFileCache(null);
                    want.ncfile.close();
                    log.debug("close " + want.ncfile.getLocation());
                }
                catch (IOException e) {
                    log.error("close failed on " + want.ncfile.getLocation(), (Throwable)e);
                }
                want.ncfile = null;
            }
            wantCacheElem.list.clear();
        }
        this.cache.remove(hashKey);
    }

    @Override
    public boolean release(FileCacheable ncfile) throws IOException {
        if (ncfile == null) {
            return false;
        }
        if (this.disabled.get()) {
            ncfile.setFileCache(null);
            ncfile.close();
            return false;
        }
        CacheElement.CacheFile file = this.files.get(ncfile);
        if (file != null) {
            if (!file.isLocked.get()) {
                cacheLog.warn("FileCache " + this.name + " release " + ncfile.getLocation() + " not locked; hash= " + ncfile.hashCode());
            }
            file.lastAccessed = System.currentTimeMillis();
            ++file.countAccessed;
            try {
                file.ncfile.release();
                file.isLocked.set(false);
            }
            catch (IOException ioe) {
                cacheLog.error("FileCache {} release failed on {} - will try to remove from cache. Failure due to:", new Object[]{this.name, file.getCacheName(), ioe});
                this.remove(file);
            }
            if (cacheLog.isDebugEnabled()) {
                cacheLog.debug("FileCache " + this.name + " release " + ncfile.getLocation() + "; hash= " + ncfile.hashCode());
            }
            return true;
        }
        return false;
    }

    private int countLocked() {
        int count = 0;
        for (CacheElement.CacheFile file : this.files.values()) {
            if (!file.isLocked.get()) continue;
            ++count;
        }
        return count;
    }

    public String getInfo(FileCacheable ncfile) {
        if (ncfile == null) {
            return "";
        }
        CacheElement.CacheFile file = this.files.get(ncfile);
        if (file != null) {
            return "File is in cache= " + file;
        }
        return "File not in cache";
    }

    Map<Object, CacheElement> getCache() {
        return this.cache;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public synchronized void clearCache(boolean force) {
        ArrayList<Object> deleteList = new ArrayList<Object>(2 * this.cache.size());
        if (force) {
            this.cache.clear();
            deleteList.addAll(this.files.values());
            this.files.clear();
        } else {
            Object file;
            Iterator<CacheElement.CacheFile> iter = this.files.values().iterator();
            while (iter.hasNext()) {
                file = iter.next();
                if (!((CacheElement.CacheFile)file).isLocked.compareAndSet(false, true)) continue;
                ((CacheElement.CacheFile)file).remove();
                deleteList.add(file);
                iter.remove();
            }
            file = this.cache;
            synchronized (file) {
                Iterator<CacheElement> iterator = this.cache.values().iterator();
                while (iterator.hasNext()) {
                    CacheElement elem;
                    CacheElement cacheElement = elem = iterator.next();
                    synchronized (cacheElement) {
                        if (elem.list.isEmpty()) {
                            this.cache.remove(elem.hashKey);
                        }
                    }
                }
            }
        }
        for (Object file : deleteList) {
            if (force && ((CacheElement.CacheFile)file).isLocked.get()) {
                cacheLog.warn("FileCache " + this.name + " force close locked file= " + file);
            }
            try {
                if (file == null || ((CacheElement.CacheFile)file).ncfile == null) {
                    log.error(String.format("FileCache %s: null file or null ncfile", this.name));
                } else {
                    ((CacheElement.CacheFile)file).ncfile.setFileCache(null);
                    ((CacheElement.CacheFile)file).ncfile.close();
                }
                if (file == null) continue;
                ((CacheElement.CacheFile)file).ncfile = null;
            }
            catch (IOException e) {
                log.error("FileCache " + this.name + " close failed on " + file);
            }
        }
        if (cacheLog.isDebugEnabled()) {
            cacheLog.debug("*FileCache " + this.name + " clearCache force= " + force + " deleted= " + deleteList.size() + " left=" + this.files.size());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void showCache(Formatter format) {
        ArrayList<CacheElement.CacheFile> allFiles = new ArrayList<CacheElement.CacheFile>(this.files.size());
        Iterator<CacheElement> iterator = this.cache.values().iterator();
        while (iterator.hasNext()) {
            CacheElement elem;
            CacheElement cacheElement = elem = iterator.next();
            synchronized (cacheElement) {
                allFiles.addAll(elem.list);
            }
        }
        Collections.sort(allFiles);
        format.format("%nFileCache %s (min=%d softLimit=%d hardLimit=%d scour=%d secs):%n", this.name, this.minElements, this.softLimit, this.hardLimit, this.period / 1000L);
        format.format(" isLocked  accesses lastAccess                   location %n", new Object[0]);
        for (CacheElement.CacheFile file : allFiles) {
            String loc = file.ncfile != null ? file.ncfile.getLocation() : "null";
            format.format("%8s %9d %s == %s %n", file.isLocked, file.countAccessed, CalendarDateFormatter.toDateTimeStringISO(file.lastAccessed), loc);
        }
        this.showStats(format);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public List<String> showCache() {
        ArrayList<CacheElement.CacheFile> allFiles = new ArrayList<CacheElement.CacheFile>(this.files.size());
        Iterator<CacheElement> iterator = this.cache.values().iterator();
        while (iterator.hasNext()) {
            CacheElement elem;
            CacheElement cacheElement = elem = iterator.next();
            synchronized (cacheElement) {
                allFiles.addAll(elem.list);
            }
        }
        Collections.sort(allFiles);
        ArrayList<String> result = new ArrayList<String>(allFiles.size());
        for (CacheElement.CacheFile file : allFiles) {
            result.add(file.toString());
        }
        return result;
    }

    @Override
    public void showStats(Formatter format) {
        format.format("  hits= %d miss= %d nfiles= %d elems= %d%n", this.hits.get(), this.miss.get(), this.files.size(), this.cache.values().size());
    }

    @Override
    public void showTracking(Formatter format) {
        if (this.track == null) {
            return;
        }
        ArrayList<Tracker> all = new ArrayList<Tracker>(this.track.size());
        all.addAll(this.track.values());
        Collections.sort(all);
        int seq = 0;
        int countAll = 0;
        int countHits = 0;
        int countMiss = 0;
        format.format("%nTracking All files in cache %s%n", this.name);
        format.format("    #    accum       hit    miss  file%n", new Object[0]);
        for (Tracker t : all) {
            countHits += t.hit;
            countMiss += t.miss;
            format.format("%6d  %7d : %6d %6d %s%n", ++seq, countAll += t.hit + t.miss, t.hit, t.miss, t.key);
        }
        float r = countAll == 0 ? 0.0f : (float)countHits / (float)countAll;
        format.format("  total=%7d : %6d %6d hit ratio=%f%n", countAll, countHits, countMiss, Float.valueOf(r));
    }

    @Override
    public void resetTracking() {
        this.track = new ConcurrentHashMap(5000);
        this.trackAll = true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    synchronized void cleanup(int maxElements) {
        try {
            Object file;
            int size = this.files.size();
            if (size <= this.minElements) {
                return;
            }
            if (cacheLog.isDebugEnabled()) {
                cacheLog.debug("FileCache {} cleanup started at {} for maxElements={}", new Object[]{this.name, CalendarDate.present(), maxElements});
            }
            this.cleanups.incrementAndGet();
            ArrayList<CacheFileSorter> unlockedFiles = new ArrayList<CacheFileSorter>();
            for (CacheElement.CacheFile file2 : this.files.values()) {
                if (file2.isLocked.get()) continue;
                unlockedFiles.add(new CacheFileSorter(file2));
            }
            Collections.sort(unlockedFiles);
            int need2delete = size - this.minElements;
            int minDelete = size - maxElements;
            ArrayList<Object> deleteList = new ArrayList<Object>(need2delete);
            int count = 0;
            Iterator iter = unlockedFiles.iterator();
            while (iter.hasNext() && count < need2delete) {
                file = ((CacheFileSorter)iter.next()).cacheFile;
                if (!((CacheElement.CacheFile)file).isLocked.compareAndSet(false, true)) continue;
                ((CacheElement.CacheFile)file).remove();
                deleteList.add(file);
                ++count;
            }
            if (count < minDelete) {
                cacheLog.warn("FileCache " + this.name + " cleanup couldnt remove enough to keep under the maximum= " + maxElements + " due to locked files; currently at = " + (size - count));
            }
            file = this.cache;
            synchronized (file) {
                Iterator<CacheElement> iterator = this.cache.values().iterator();
                while (iterator.hasNext()) {
                    CacheElement cacheElement;
                    CacheElement cacheElement2 = cacheElement = iterator.next();
                    synchronized (cacheElement2) {
                        if (cacheElement.list.isEmpty()) {
                            this.cache.remove(cacheElement.hashKey);
                        }
                    }
                }
            }
            long start = System.currentTimeMillis();
            for (CacheElement.CacheFile cacheFile : deleteList) {
                if (null == this.files.remove(cacheFile.ncfile) && cacheLog.isDebugEnabled()) {
                    cacheLog.debug(" FileCache {} cleanup failed to remove {}%n", (Object)this.name, (Object)cacheFile.ncfile.getLocation());
                }
                try {
                    cacheFile.ncfile.setFileCache(null);
                    cacheFile.ncfile.close();
                    cacheFile.ncfile = null;
                }
                catch (IOException e) {
                    log.error("FileCache " + this.name + " close failed on " + cacheFile.getCacheName());
                }
            }
            long l = System.currentTimeMillis() - start;
            if (cacheLog.isDebugEnabled()) {
                cacheLog.debug(" FileCache {} cleanup had={} removed={} took={} msecs%n", new Object[]{this.name, size, deleteList.size(), l});
            }
        }
        finally {
            this.hasScheduled.set(false);
        }
    }

    static {
        lock = new Object();
    }

    private class CleanupTask
    extends TimerTask {
        private CleanupTask() {
        }

        @Override
        public void run() {
            if (FileCache.this.disabled.get()) {
                return;
            }
            FileCache.this.cleanup(FileCache.this.softLimit);
        }
    }

    private class CacheFileSorter
    implements Comparable<CacheFileSorter> {
        private final CacheElement.CacheFile cacheFile;
        private final long lastAccessed;

        CacheFileSorter(CacheElement.CacheFile cacheFile) {
            this.cacheFile = cacheFile;
            this.lastAccessed = cacheFile.lastAccessed;
        }

        @Override
        public int compareTo(CacheFileSorter o) {
            return Long.compare(this.lastAccessed, o.lastAccessed);
        }
    }

    class CacheElement {
        @GuardedBy(value="this")
        final List<CacheFile> list = new LinkedList<CacheFile>();
        final Object hashKey;

        CacheElement(FileCacheable ncfile, Object hashKey) {
            this.hashKey = hashKey;
            CacheFile file = new CacheFile(ncfile);
            this.list.add(file);
            FileCache.this.files.put(ncfile, file);
            if (cacheLog.isDebugEnabled()) {
                cacheLog.debug("CacheElement add to cache " + hashKey + " " + FileCache.this.name);
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        CacheFile addFile(FileCacheable ncfile) {
            CacheFile file = new CacheFile(ncfile);
            CacheElement cacheElement = this;
            synchronized (cacheElement) {
                this.list.add(file);
            }
            FileCache.this.files.put(ncfile, file);
            return file;
        }

        public String toString() {
            return this.hashKey + " count=" + this.list.size();
        }

        class CacheFile
        implements Comparable<CacheFile> {
            FileCacheable ncfile;
            final AtomicBoolean isLocked = new AtomicBoolean(true);
            int countAccessed;
            long lastModified;
            long lastAccessed;

            private CacheFile(FileCacheable ncfile) {
                this.ncfile = ncfile;
                this.lastModified = ncfile.getLastModified();
                this.lastAccessed = System.currentTimeMillis();
                ncfile.setFileCache(FileCache.this);
                if (cacheLog.isDebugEnabled()) {
                    cacheLog.debug("FileCache " + FileCache.this.name + " add to cache " + CacheElement.this.hashKey);
                }
            }

            String getCacheName() {
                return this.ncfile.getLocation();
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            void remove() {
                CacheElement cacheElement = CacheElement.this;
                synchronized (cacheElement) {
                    if (!CacheElement.this.list.remove(this)) {
                        cacheLog.warn("FileCache " + FileCache.this.name + " could not remove " + this.ncfile.getLocation());
                    }
                }
                if (cacheLog.isDebugEnabled()) {
                    cacheLog.debug("FileCache " + FileCache.this.name + " remove " + this.ncfile.getLocation());
                }
            }

            public String toString() {
                String name = this.ncfile == null ? "ncfile is null" : this.ncfile.getLocation();
                return this.isLocked + " " + this.countAccessed + " " + CalendarDateFormatter.toDateTimeStringISO(this.lastAccessed) + "   " + name;
            }

            @Override
            public int compareTo(CacheFile o) {
                return Long.compare(this.lastAccessed, o.lastAccessed);
            }
        }
    }

    private static class Tracker
    implements Comparable<Tracker> {
        Object key;
        int hit;
        int miss;

        private Tracker(Object key) {
            this.key = key;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            Tracker tracker = (Tracker)o;
            return this.key.equals(tracker.key);
        }

        public int hashCode() {
            return this.key.hashCode();
        }

        @Override
        public int compareTo(Tracker o) {
            return Integer.compare(this.hit + this.miss, o.hit + o.miss);
        }
    }
}

