/*
 * Decompiled with CFR 0.152.
 */
package ucar.nc2.internal.grid;

import com.google.common.base.Preconditions;
import com.google.common.math.DoubleMath;
import java.util.Arrays;
import java.util.Formatter;
import java.util.Optional;
import ucar.array.InvalidRangeException;
import ucar.array.MinMax;
import ucar.array.Range;
import ucar.nc2.constants.AxisType;
import ucar.nc2.grid.CoordInterval;
import ucar.nc2.grid.GridAxis;
import ucar.nc2.grid.GridAxis1D;
import ucar.nc2.grid.GridAxis1DTime;
import ucar.nc2.time.CalendarDate;
import ucar.nc2.time.CalendarDateRange;
import ucar.nc2.util.Misc;

public class GridAxis1DHelper {
    private final GridAxis1D orgGridAxis;

    public GridAxis1DHelper(GridAxis1D orgGrid) {
        this.orgGridAxis = orgGrid;
    }

    public int findCoordElement(CoordInterval target, boolean bounded) {
        switch (this.orgGridAxis.getSpacing()) {
            case regularInterval: {
                return this.findCoordElementRegular(target.midpoint(), bounded);
            }
            case contiguousInterval: {
                return this.findCoordElementContiguous(target.midpoint(), bounded);
            }
            case discontiguousInterval: {
                return this.findCoordElementDiscontiguousInterval(target, bounded);
            }
        }
        throw new IllegalStateException("unknown spacing" + (Object)((Object)this.orgGridAxis.getSpacing()));
    }

    public int findCoordElement(double target, boolean bounded) {
        switch (this.orgGridAxis.getSpacing()) {
            case regularInterval: 
            case regularPoint: {
                return this.findCoordElementRegular(target, bounded);
            }
            case contiguousInterval: 
            case irregularPoint: {
                return this.findCoordElementContiguous(target, bounded);
            }
            case discontiguousInterval: {
                return this.findCoordElementDiscontiguousInterval(target, bounded);
            }
        }
        throw new IllegalStateException("unknown spacing" + (Object)((Object)this.orgGridAxis.getSpacing()));
    }

    private int findCoordElementRegular(double coordValue, boolean bounded) {
        int n = this.orgGridAxis.getNcoords();
        if (n == 1 && bounded) {
            return 0;
        }
        double distance = coordValue - this.orgGridAxis.getCoordEdge1(0);
        double exactNumSteps = distance / this.orgGridAxis.getResolution();
        int index = (int)exactNumSteps;
        if (bounded && index < 0) {
            return 0;
        }
        if (bounded && index >= n) {
            return n - 1;
        }
        if (index >= 0 && index < n) {
            double lower = this.orgGridAxis.getCoordEdge1(index);
            double upper = this.orgGridAxis.getCoordEdge2(index);
            if (this.orgGridAxis.isAscending()) {
                assert (lower <= coordValue) : lower + " should be le " + coordValue;
                assert (upper >= coordValue) : upper + " should be ge " + coordValue;
            } else {
                assert (lower >= coordValue) : lower + " should be ge " + coordValue;
                assert (upper <= coordValue) : upper + " should be le " + coordValue;
            }
        }
        return index;
    }

    private int findCoordElementContiguous(double target, boolean bounded) {
        int n = this.orgGridAxis.getNcoords();
        MinMax minmax = this.orgGridAxis.getCoordEdgeMinMax();
        if (target < minmax.min()) {
            return bounded ? 0 : -1;
        }
        if (target > minmax.max()) {
            return bounded ? n - 1 : n;
        }
        int low = 0;
        int high = n - 1;
        if (this.orgGridAxis.isAscending()) {
            while (high > low + 1) {
                int mid = (low + high) / 2;
                if (this.intervalContains(target, mid, true, false)) {
                    return mid;
                }
                if (this.orgGridAxis.getCoordEdge2(mid) < target) {
                    low = mid;
                    continue;
                }
                high = mid;
            }
            return this.intervalContains(target, low, true, false) ? low : high;
        }
        while (high > low + 1) {
            int mid = (low + high) / 2;
            if (this.intervalContains(target, mid, false, false)) {
                return mid;
            }
            if (this.orgGridAxis.getCoordEdge2(mid) < target) {
                high = mid;
                continue;
            }
            low = mid;
        }
        return this.intervalContains(target, low, false, false) ? low : high;
    }

    private int findCoordElementDiscontiguousInterval(CoordInterval target, boolean bounded) {
        Preconditions.checkNotNull((Object)target);
        int n = this.orgGridAxis.getNcoords();
        double lowerValue = Math.min(target.start(), target.end());
        double upperValue = Math.max(target.start(), target.end());
        MinMax minmax = this.orgGridAxis.getCoordEdgeMinMax();
        if (upperValue < minmax.min()) {
            return bounded ? 0 : -1;
        }
        if (lowerValue > minmax.max()) {
            return bounded ? n - 1 : n;
        }
        int index = -1;
        for (int i = 0; i < this.orgGridAxis.getNcoords(); ++i) {
            if (!target.fuzzyEquals(this.orgGridAxis.getCoordInterval(i), 1.0E-8)) continue;
            return i;
        }
        if (bounded) {
            index = this.findCoordElementDiscontiguousInterval(target.midpoint(), bounded);
        }
        return index;
    }

    private int findCoordElementDiscontiguousInterval(double target, boolean bounded) {
        int idx = this.findSingleHit(target);
        if (idx >= 0) {
            return idx;
        }
        if (idx == -1) {
            return -1;
        }
        return this.findClosestDiscontiguousInterval(target);
    }

    private boolean intervalContains(double target, int coordIdx) {
        double midVal2;
        double midVal1 = this.orgGridAxis.getCoordEdge1(coordIdx);
        boolean ascending = midVal1 < (midVal2 = this.orgGridAxis.getCoordEdge2(coordIdx));
        return this.intervalContains(target, coordIdx, ascending, true);
    }

    private boolean intervalContains(double target, int coordIdx, boolean ascending, boolean closed) {
        double upperVal;
        double lowerVal = ascending ? this.orgGridAxis.getCoordEdge1(coordIdx) : this.orgGridAxis.getCoordEdge2(coordIdx);
        double d = upperVal = ascending ? this.orgGridAxis.getCoordEdge2(coordIdx) : this.orgGridAxis.getCoordEdge1(coordIdx);
        if (closed) {
            return lowerVal <= target && target <= upperVal;
        }
        return lowerVal <= target && target < upperVal;
    }

    private int findSingleHit(double target) {
        int hits = 0;
        int idxFound = -1;
        int n = this.orgGridAxis.getNcoords();
        for (int i = 0; i < n; ++i) {
            if (!this.intervalContains(target, i)) continue;
            ++hits;
            idxFound = i;
        }
        if (hits == 1) {
            return idxFound;
        }
        if (hits == 0) {
            return -1;
        }
        return -hits;
    }

    private int findClosestDiscontiguousInterval(double target) {
        boolean isTemporal = this.orgGridAxis.getAxisType().equals((Object)AxisType.Time);
        return isTemporal ? this.findClosestDiscontiguousTimeInterval(target) : this.findClosestDiscontiguousNonTimeInterval(target);
    }

    private int findClosestDiscontiguousTimeInterval(double target) {
        double minDiff = Double.MAX_VALUE;
        double useValue = Double.MIN_VALUE;
        int idxFound = -1;
        for (int i = 0; i < this.orgGridAxis.getNcoords(); ++i) {
            boolean tiebreaker;
            double coord;
            if (!this.intervalContains(target, i) || !((coord = this.orgGridAxis.getCoordEdge2(i)) >= target)) continue;
            double width = this.orgGridAxis.getCoordEdge2(i) - this.orgGridAxis.getCoordEdge1(i);
            double diff = Math.abs(coord - target);
            boolean bl = tiebreaker = diff == minDiff && width != 0.0 && width < useValue;
            if (!(diff < minDiff) && !tiebreaker) continue;
            minDiff = diff;
            idxFound = i;
            useValue = width;
        }
        return idxFound;
    }

    private int findClosestDiscontiguousNonTimeInterval(double target) {
        int idxFound = -1;
        double minDiff = Double.MAX_VALUE;
        double useValue = Double.MIN_VALUE;
        for (int i = 0; i < this.orgGridAxis.getNcoords(); ++i) {
            boolean tiebreaker;
            if (!this.intervalContains(target, i)) continue;
            double coord = this.orgGridAxis.getCoordMidpoint(i);
            double diff = Math.abs(coord - target);
            boolean bl = tiebreaker = diff == minDiff && coord > useValue;
            if (!(diff < minDiff) && !tiebreaker) continue;
            minDiff = diff;
            idxFound = i;
            useValue = coord;
        }
        return idxFound;
    }

    public Optional<GridAxis1D.Builder<?>> subset(double minValue, double maxValue, int stride, Formatter errLog) {
        return this.makeSubsetValues(minValue, maxValue, stride, errLog);
    }

    public GridAxis1D.Builder<?> subsetClosest(double want) {
        return this.makeSubsetValuesClosest(want);
    }

    public GridAxis1D.Builder<?> subsetClosest(CoordInterval want) {
        for (int idx = 0; idx < this.orgGridAxis.getNcoords(); ++idx) {
            double bound1 = this.orgGridAxis.getCoordEdge1(idx);
            double bound2 = this.orgGridAxis.getCoordEdge2(idx);
            if (!Misc.nearlyEquals(bound1, want.start()) || !Misc.nearlyEquals(bound2, want.end())) continue;
            return this.makeSubsetByIndex(Range.make(idx, idx));
        }
        return this.makeSubsetValuesClosest(want);
    }

    public GridAxis1D.Builder<?> subsetLatest() {
        return this.makeSubsetValuesLatest();
    }

    public GridAxis1D.Builder<?> subsetClosest(CalendarDate date) {
        double want = ((GridAxis1DTime)this.orgGridAxis).makeValue(date);
        return this.isDiscontiguousInterval() ? this.subsetClosestDiscontiguousInterval(date) : this.makeSubsetValuesClosest(want);
    }

    private GridAxis1D.Builder<?> subsetClosestDiscontiguousInterval(CalendarDate date) {
        double target = ((GridAxis1DTime)this.orgGridAxis).makeValue(date);
        double maxDateToSearch = 0.0;
        int intervals = 0;
        for (int i = 0; i < this.orgGridAxis.getNcoords(); ++i) {
            double bound2;
            if (!this.intervalContains(target, i)) continue;
            ++intervals;
            double bound1 = this.orgGridAxis.getCoordEdge1(i);
            double intervalEnd = Math.max(bound1, bound2 = this.orgGridAxis.getCoordEdge2(i));
            if (!(intervalEnd > maxDateToSearch)) continue;
            maxDateToSearch = intervalEnd;
        }
        Optional<GridAxis1D.Builder<Object>> multipleIntervalBuilder = Optional.empty();
        if (intervals > 0) {
            multipleIntervalBuilder = this.subset(target, maxDateToSearch, 1, new Formatter());
        }
        return multipleIntervalBuilder.orElseGet(() -> this.makeSubsetValuesClosest(((GridAxis1DTime)this.orgGridAxis).makeValue(date)));
    }

    public GridAxis1D.Builder<?> subsetClosest(CalendarDate[] date) {
        double start = ((GridAxis1DTime)this.orgGridAxis).makeValue(date[0]);
        double end = ((GridAxis1DTime)this.orgGridAxis).makeValue(date[1]);
        return this.makeSubsetValuesClosest(CoordInterval.create(start, end));
    }

    public Optional<GridAxis1D.Builder<?>> subset(CalendarDateRange dateRange, int stride, Formatter errLog) {
        double min = ((GridAxis1DTime)this.orgGridAxis).makeValue(dateRange.getStart());
        double max = ((GridAxis1DTime)this.orgGridAxis).makeValue(dateRange.getEnd());
        return this.makeSubsetValues(min, max, stride, errLog);
    }

    public GridAxis1D.Builder<?> makeSubsetByIndex(Range subsetRange) {
        int ncoords = subsetRange.length();
        Preconditions.checkArgument((subsetRange.last() < this.orgGridAxis.getNcoords() ? 1 : 0) != 0);
        double resolution = 0.0;
        if (this.orgGridAxis.getSpacing().isRegular()) {
            resolution = (double)subsetRange.stride() * this.orgGridAxis.getResolution();
        }
        GridAxis1D.Builder<?> builder = this.orgGridAxis.toBuilder();
        builder.subset(ncoords, this.orgGridAxis.getCoordMidpoint(subsetRange.first()), this.orgGridAxis.getCoordMidpoint(subsetRange.last()), resolution, subsetRange);
        return builder;
    }

    private GridAxis1D.Builder<?> makeSubsetValuesClosest(CoordInterval want) {
        int closest_index = this.findCoordElement(want, true);
        Range range = Range.make(closest_index, closest_index);
        GridAxis1D.Builder<?> builder = this.orgGridAxis.toBuilder();
        if (this.orgGridAxis.getSpacing() == GridAxis.Spacing.regularInterval) {
            double val1 = this.orgGridAxis.getCoordEdge1(closest_index);
            double val2 = this.orgGridAxis.getCoordEdge2(closest_index);
            builder.subset(1, val1, val2, val2 - val1, range);
        } else {
            builder.subset(1, 0.0, 0.0, 0.0, range);
        }
        return builder;
    }

    private GridAxis1D.Builder<?> makeSubsetValuesClosest(double want) {
        Range range;
        int closest_index = this.findCoordElement(want, true);
        GridAxis1D.Builder<?> builder = this.orgGridAxis.toBuilder();
        try {
            range = new Range(closest_index, closest_index);
        }
        catch (InvalidRangeException e) {
            throw new RuntimeException(e);
        }
        if (this.orgGridAxis.getSpacing() == GridAxis.Spacing.regularPoint) {
            double val = this.orgGridAxis.getCoordMidpoint(closest_index);
            builder.subset(1, val, val, 0.0, range);
        } else if (this.orgGridAxis.getSpacing() == GridAxis.Spacing.regularInterval) {
            double val1 = this.orgGridAxis.getCoordEdge1(closest_index);
            double val2 = this.orgGridAxis.getCoordEdge2(closest_index);
            builder.subset(1, val1, val2, val2 - val1, range);
        } else {
            builder.subset(1, 0.0, 0.0, 0.0, range);
        }
        return builder;
    }

    private Optional<GridAxis1D.Builder<?>> makeSubsetValues(double minValue, double maxValue, int stride, Formatter errLog) {
        int count;
        double lower = this.orgGridAxis.isAscending() ? Math.min(minValue, maxValue) : Math.max(minValue, maxValue);
        double upper = this.orgGridAxis.isAscending() ? Math.max(minValue, maxValue) : Math.min(minValue, maxValue);
        int minIndex = this.findCoordElement(lower, false);
        int maxIndex = this.findCoordElement(upper, false);
        if (minIndex >= this.orgGridAxis.getNcoords()) {
            errLog.format("no points in subset: lower %f > end %f", lower, this.orgGridAxis.getEndValue());
            return Optional.empty();
        }
        if (maxIndex < 0) {
            errLog.format("no points in subset: upper %f < start %f", upper, this.orgGridAxis.getStartValue());
            return Optional.empty();
        }
        if (minIndex < 0) {
            minIndex = 0;
        }
        if (maxIndex >= this.orgGridAxis.getNcoords()) {
            maxIndex = this.orgGridAxis.getNcoords() - 1;
        }
        if ((count = maxIndex - minIndex + 1) <= 0) {
            throw new IllegalArgumentException("no points in subset");
        }
        try {
            return Optional.of(this.makeSubsetByIndex(new Range(minIndex, maxIndex, stride)));
        }
        catch (InvalidRangeException e) {
            errLog.format("%s", e.getMessage());
            return Optional.empty();
        }
    }

    private GridAxis1D.Builder<?> makeSubsetValuesLatest() {
        Range range;
        int last = this.orgGridAxis.getNcoords() - 1;
        double val = this.orgGridAxis.getCoordMidpoint(last);
        try {
            range = new Range(last, last);
        }
        catch (InvalidRangeException e) {
            throw new RuntimeException(e);
        }
        GridAxis1D.Builder<?> builder = this.orgGridAxis.toBuilder();
        builder.subset(1, val, val, 0.0, range);
        return builder;
    }

    int search(double want) {
        if (this.orgGridAxis.getNcoords() == 1) {
            return DoubleMath.fuzzyEquals((double)want, (double)this.orgGridAxis.getStartValue(), (double)1.0E-8) ? 0 : -1;
        }
        if (this.orgGridAxis.isRegular()) {
            double ival;
            double fval = (want - this.orgGridAxis.getStartValue()) / this.orgGridAxis.getResolution();
            return DoubleMath.fuzzyEquals((double)fval, (double)(ival = Math.rint(fval)), (double)1.0E-8) ? (int)ival : (int)(-ival) - 1;
        }
        return Arrays.binarySearch(this.orgGridAxis.getValues(), want);
    }

    private boolean isDiscontiguousInterval() {
        GridAxis.Spacing spacing = this.orgGridAxis.getSpacing();
        if (spacing != null) {
            return spacing.equals((Object)GridAxis.Spacing.discontiguousInterval);
        }
        return false;
    }
}

