/*
 * Decompiled with CFR 0.152.
 */
package ucar.nc2.internal.grid;

import java.util.ArrayList;
import java.util.Formatter;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import javax.annotation.Nullable;
import ucar.array.InvalidRangeException;
import ucar.array.MinMax;
import ucar.array.Range;
import ucar.nc2.constants.AxisType;
import ucar.nc2.grid.GridAxis;
import ucar.nc2.grid.GridAxis1D;
import ucar.nc2.grid.GridAxis2D;
import ucar.nc2.grid.GridHorizCoordinateSystem;
import ucar.nc2.grid.GridSubset;
import ucar.nc2.internal.grid.GridAxis1DHelper;
import ucar.nc2.internal.grid.GridLatLon2D;
import ucar.unidata.geoloc.LatLonPoint;
import ucar.unidata.geoloc.LatLonPoints;
import ucar.unidata.geoloc.LatLonRect;
import ucar.unidata.geoloc.Projection;
import ucar.unidata.geoloc.ProjectionPoint;
import ucar.unidata.geoloc.ProjectionRect;
import ucar.unidata.geoloc.projection.LatLonProjection;
import ucar.unidata.geoloc.projection.sat.Geostationary;
import ucar.unidata.geoloc.projection.sat.MSGnavigation;
import ucar.unidata.geoloc.projection.sat.VerticalPerspectiveView;

public class GridHorizCS
implements GridHorizCoordinateSystem {
    @Nullable
    private final GridAxis1D xaxis;
    @Nullable
    private final GridAxis1D yaxis;
    private final Projection projection;
    @Nullable
    private final String horizStaggerType;
    private LatLonRect llbb;
    ProjectionRect mapArea;

    public static GridHorizCS create(GridAxis xaxis, GridAxis yaxis, @Nullable Projection projection) {
        String horizStaggerType = xaxis.attributes().findAttributeString("_CoordinateStagger", null);
        if (xaxis instanceof GridAxis1D && yaxis instanceof GridAxis1D) {
            return new GridHorizCS((GridAxis1D)xaxis, (GridAxis1D)yaxis, projection, horizStaggerType);
        }
        if (xaxis instanceof GridAxis2D && yaxis instanceof GridAxis2D) {
            return new GridLatLon2D((GridAxis2D)xaxis, (GridAxis2D)yaxis, projection, horizStaggerType);
        }
        throw new RuntimeException();
    }

    GridHorizCS(@Nullable GridAxis1D xaxis, @Nullable GridAxis1D yaxis, @Nullable Projection projection, @Nullable String horizStaggerType) {
        this.xaxis = xaxis;
        this.yaxis = yaxis;
        this.projection = projection == null ? new LatLonProjection() : projection;
        this.horizStaggerType = horizStaggerType;
    }

    @Override
    @Nullable
    public GridAxis1D getXHorizAxis() {
        return this.xaxis;
    }

    @Override
    @Nullable
    public GridAxis1D getYHorizAxis() {
        return this.yaxis;
    }

    @Override
    public Projection getProjection() {
        return this.projection;
    }

    @Override
    public boolean isLatLon() {
        return this.projection.isLatLon();
    }

    @Override
    public boolean isGlobalLon() {
        if (!this.isLatLon()) {
            return false;
        }
        LatLonRect rect = this.getLatLonBoundingBox();
        return rect.getWidth() >= 360.0;
    }

    @Override
    public boolean isRegular() {
        if (!this.isRegularSpatial(this.getXHorizAxis())) {
            return false;
        }
        return this.isRegularSpatial(this.getYHorizAxis());
    }

    private boolean isRegularSpatial(GridAxis1D axis) {
        if (axis == null) {
            return false;
        }
        return axis.isRegular();
    }

    @Override
    @Nullable
    public String getGeoUnits() {
        return this.isLatLon() ? null : this.xaxis.getUnits();
    }

    @Override
    @Nullable
    public String getHorizStaggerType() {
        return this.horizStaggerType;
    }

    @Override
    public LatLonRect getLatLonBoundingBox() {
        if (this.llbb == null) {
            GridAxis1D xaxis = this.getXHorizAxis();
            GridAxis1D yaxis = this.getYHorizAxis();
            if (this.isLatLon()) {
                double startLat = yaxis.getCoordEdge1(0);
                double startLon = xaxis.getCoordEdge1(0);
                double endLat = yaxis.getCoordEdge2(yaxis.getNcoords() - 1);
                double endLon = xaxis.getCoordEdge2(xaxis.getNcoords() - 1);
                LatLonPoint llpt = LatLonPoint.create(startLat, startLon);
                this.llbb = new LatLonRect(llpt, endLat - startLat, endLon - startLon);
            } else {
                Projection dataProjection = this.getProjection();
                ProjectionRect bb = this.getBoundingBox();
                if (dataProjection != null && bb != null) {
                    this.llbb = dataProjection.projToLatLonBB(bb);
                }
            }
        }
        return this.llbb;
    }

    @Override
    public ProjectionRect getBoundingBox() {
        if (this.mapArea == null) {
            this.mapArea = new ProjectionRect(this.xaxis.getCoordEdge1(0), this.yaxis.getCoordEdge1(0), this.xaxis.getCoordEdge2(this.xaxis.getNcoords() - 1), this.yaxis.getCoordEdge2(this.yaxis.getNcoords() - 1));
        }
        return this.mapArea;
    }

    public LatLonPoint getLatLon(int xindex, int yindex) {
        double x = this.xaxis.getCoordMidpoint(xindex);
        double y = this.yaxis.getCoordMidpoint(xindex);
        return this.isLatLon() ? LatLonPoint.create(y, x) : this.getLatLon(x, y);
    }

    private LatLonPoint getLatLon(double xcoord, double ycoord) {
        Projection dataProjection = this.getProjection();
        return dataProjection.projToLatLon(ProjectionPoint.create(xcoord, ycoord));
    }

    @Override
    public Optional<GridHorizCoordinateSystem.CoordReturn> findXYindexFromCoord(double x, double y) {
        GridAxis1DHelper xhelper = new GridAxis1DHelper(this.xaxis);
        GridAxis1DHelper yhelper = new GridAxis1DHelper(this.yaxis);
        GridHorizCoordinateSystem.CoordReturn result = new GridHorizCoordinateSystem.CoordReturn();
        if (this.xaxis.getAxisType() == AxisType.Lon) {
            x = LatLonPoints.lonNormalFrom(x, this.xaxis.getStartValue());
        }
        result.xindex = xhelper.findCoordElement(x, false);
        result.yindex = yhelper.findCoordElement(y, false);
        if (result.xindex >= 0 && result.xindex < this.xaxis.getNcoords() && result.yindex >= 0 && result.yindex < this.yaxis.getNcoords()) {
            result.xcoord = this.xaxis.getCoordMidpoint(result.xindex);
            result.ycoord = this.yaxis.getCoordMidpoint(result.yindex);
            return Optional.of(result);
        }
        return Optional.empty();
    }

    @Override
    public List<GridAxis> subset(GridSubset params, Formatter errlog) {
        ArrayList<GridAxis> result = new ArrayList<GridAxis>();
        Integer horizStride = params.getHorizStride();
        if (horizStride == null || horizStride < 1) {
            horizStride = 1;
        }
        LatLonRect llbb = params.getLatLonBoundingBox();
        ProjectionRect projbb = params.getProjectionBoundingBox();
        if (projbb != null) {
            GridAxis1DHelper yhelper = new GridAxis1DHelper(this.yaxis);
            yhelper.subset(projbb.getMinY(), projbb.getMaxY(), horizStride, errlog).ifPresent(b -> result.add(b.build()));
            GridAxis1DHelper xhelper = new GridAxis1DHelper(this.xaxis);
            xhelper.subset(projbb.getMinX(), projbb.getMaxX(), horizStride, errlog).ifPresent(b -> result.add(b.build()));
        } else if (llbb != null && this.isLatLon()) {
            GridAxis1DHelper yhelper = new GridAxis1DHelper(this.yaxis);
            yhelper.subset(projbb.getMinY(), projbb.getMaxY(), horizStride, errlog).ifPresent(b -> result.add(b.build()));
            GridAxis1DHelper xhelper = new GridAxis1DHelper(this.xaxis);
            xhelper.subset(projbb.getMinX(), projbb.getMaxX(), horizStride, errlog).ifPresent(b -> result.add(b.build()));
        } else if (horizStride > 1) {
            try {
                Range yRange = this.yaxis.getRange().copyWithStride(horizStride);
                result.add(((GridAxis1D.Builder)this.yaxis.toBuilder().setRange(yRange)).build());
                Range xRange = this.xaxis.getRange().copyWithStride(horizStride);
                result.add(((GridAxis1D.Builder)this.xaxis.toBuilder().setRange(xRange)).build());
            }
            catch (InvalidRangeException e) {
                errlog.format(e.getMessage(), new Object[0]);
            }
        } else {
            result.add(this.yaxis);
            result.add(this.xaxis);
        }
        return result;
    }

    public List<Range> getRangesFromLatLonRect(LatLonRect rect) throws InvalidRangeException {
        double maxy;
        double maxx;
        double miny;
        double minx;
        Projection proj = this.getProjection();
        if (!(proj == null || proj instanceof VerticalPerspectiveView || proj instanceof MSGnavigation || proj instanceof Geostationary)) {
            LatLonRect bb = this.getLatLonBoundingBox();
            LatLonRect rect2 = bb.intersect(rect);
            if (null == rect2) {
                throw new InvalidRangeException("Request Bounding box does not intersect Grid ");
            }
            rect = rect2;
        }
        if (this.isLatLon()) {
            LatLonPoint llpt = rect.getLowerLeftPoint();
            LatLonPoint urpt = rect.getUpperRightPoint();
            LatLonPoint lrpt = rect.getLowerRightPoint();
            LatLonPoint ulpt = rect.getUpperLeftPoint();
            minx = this.getMinOrMaxLon(llpt.getLongitude(), ulpt.getLongitude(), true);
            miny = Math.min(llpt.getLatitude(), lrpt.getLatitude());
            maxx = this.getMinOrMaxLon(urpt.getLongitude(), lrpt.getLongitude(), false);
            maxy = Math.min(ulpt.getLatitude(), urpt.getLatitude());
            MinMax minmaxLon = this.xaxis.getCoordEdgeMinMax();
            minx = LatLonPoints.lonNormalFrom(minx, minmaxLon.min());
            maxx = LatLonPoints.lonNormalFrom(maxx, minmaxLon.min());
        } else {
            ProjectionRect prect = this.getProjection().latLonToProjBB(rect);
            minx = prect.getMinPoint().getX();
            miny = prect.getMinPoint().getY();
            maxx = prect.getMaxPoint().getX();
            maxy = prect.getMaxPoint().getY();
        }
        GridAxis1DHelper xhelper = new GridAxis1DHelper(this.xaxis);
        GridAxis1DHelper yhelper = new GridAxis1DHelper(this.yaxis);
        int minxIndex = xhelper.findCoordElement(minx, true);
        int minyIndex = yhelper.findCoordElement(miny, true);
        int maxxIndex = xhelper.findCoordElement(maxx, true);
        int maxyIndex = yhelper.findCoordElement(maxy, true);
        ArrayList<Range> list = new ArrayList<Range>();
        list.add(new Range(Math.min(minyIndex, maxyIndex), Math.max(minyIndex, maxyIndex)));
        list.add(new Range(Math.min(minxIndex, maxxIndex), Math.max(minxIndex, maxxIndex)));
        return list;
    }

    double getMinOrMaxLon(double lon1, double lon2, boolean wantMin) {
        double midpoint = (lon1 + lon2) / 2.0;
        lon1 = LatLonPoints.lonNormal(lon1, midpoint);
        lon2 = LatLonPoints.lonNormal(lon2, midpoint);
        return wantMin ? Math.min(lon1, lon2) : Math.max(lon1, lon2);
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        GridHorizCS that = (GridHorizCS)o;
        return Objects.equals(this.xaxis, that.xaxis) && Objects.equals(this.yaxis, that.yaxis) && Objects.equals(this.projection, that.projection) && Objects.equals(this.horizStaggerType, that.horizStaggerType);
    }

    public int hashCode() {
        return Objects.hash(this.xaxis, this.yaxis, this.projection, this.horizStaggerType);
    }
}

