/*
 * Decompiled with CFR 0.152.
 */
package ucar.unidata.geoloc.projection.proj4;

import com.google.common.base.Preconditions;
import java.util.Objects;
import javax.annotation.concurrent.Immutable;
import ucar.unidata.geoloc.Earth;
import ucar.unidata.geoloc.EarthEllipsoid;
import ucar.unidata.geoloc.LatLonPoint;
import ucar.unidata.geoloc.Projection;
import ucar.unidata.geoloc.ProjectionPoint;
import ucar.unidata.geoloc.projection.AbstractProjection;
import ucar.unidata.geoloc.projection.proj4.MapMath;

@Immutable
public class CylindricalEqualAreaProjection
extends AbstractProjection {
    private final double trueScaleLatitude;
    private final double lon0;
    private final double scaleFactor;
    private final double projectionLongitude;
    private final double falseEasting;
    private final double falseNorthing;
    private final Earth earth;
    private final double e;
    private final double one_es;
    private final double totalScale;
    private final double qp;
    private final double[] apa;

    public CylindricalEqualAreaProjection() {
        this(0.0, 1.0, 0.0, 0.0, EarthEllipsoid.WGS84);
    }

    public CylindricalEqualAreaProjection(double lon0, double trueScaleLatitude, double falseEasting, double falseNorthing, Earth earth) {
        super("CylindricalEqualAreaProjection", false);
        Preconditions.checkNotNull((Object)earth, (Object)"CylindricalEqualAreaProjection constructor requires non-null Earth");
        this.lon0 = lon0;
        this.projectionLongitude = Math.toRadians(lon0);
        this.trueScaleLatitude = trueScaleLatitude;
        this.falseEasting = falseEasting;
        this.falseNorthing = falseNorthing;
        this.earth = earth;
        this.e = earth.getEccentricity();
        double es = earth.getEccentricitySquared();
        this.one_es = 1.0 - es;
        this.totalScale = earth.getMajor() * 0.001;
        double t = Math.toRadians(trueScaleLatitude);
        double cost = Math.cos(t);
        if (!earth.isSpherical()) {
            t = Math.sin(t);
            this.scaleFactor = cost / Math.sqrt(1.0 - es * t * t);
            this.apa = MapMath.authset(es);
            this.qp = MapMath.qsfn(1.0, this.e, this.one_es);
        } else {
            this.scaleFactor = cost;
            this.apa = null;
            this.qp = Double.NaN;
        }
        this.addParameter("grid_mapping_name", "albers_conical_equal_area");
        this.addParameter("longitude_of_central_meridian", lon0);
        this.addParameter("standard_parallel", trueScaleLatitude);
        if (falseEasting != 0.0 || falseNorthing != 0.0) {
            this.addParameter("false_easting", falseEasting);
            this.addParameter("false_northing", falseNorthing);
            this.addParameter("units", "km");
        }
        this.addParameter("semi_major_axis", earth.getMajor());
        this.addParameter("inverse_flattening", 1.0 / earth.getFlattening());
    }

    @Override
    public Projection constructCopy() {
        return new CylindricalEqualAreaProjection(this.lon0, this.trueScaleLatitude, this.falseEasting, this.falseNorthing, this.earth);
    }

    @Override
    public String paramsToString() {
        return null;
    }

    @Override
    public ProjectionPoint latLonToProj(LatLonPoint latlon) {
        double toY;
        double toX;
        double lam = Math.toRadians(latlon.getLongitude() - this.lon0);
        double phi = Math.toRadians(latlon.getLatitude());
        if (this.earth.isSpherical()) {
            toX = this.scaleFactor * lam;
            toY = Math.sin(phi) / this.scaleFactor;
        } else {
            toX = this.scaleFactor * lam;
            toY = 0.5 * MapMath.qsfn(Math.sin(phi), this.e, this.one_es) / this.scaleFactor;
        }
        return ProjectionPoint.create(this.totalScale * toX + this.falseEasting, this.totalScale * toY + this.falseNorthing);
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    @Override
    public LatLonPoint projToLatLon(ProjectionPoint ppt) {
        double toLon;
        double toLat;
        double x = (ppt.getX() - this.falseEasting) / this.totalScale;
        double y = (ppt.getY() - this.falseNorthing) / this.totalScale;
        if (this.earth.isSpherical()) {
            double t = Math.abs(y *= this.scaleFactor);
            if (!(t - 1.0E-10 <= 1.0)) throw new IllegalStateException();
            toLat = t >= 1.0 ? Math.toDegrees(y < 0.0 ? -1.5707963267948966 : 1.5707963267948966) : Math.toDegrees(Math.asin(y));
            toLon = Math.toDegrees(x / this.scaleFactor);
            return LatLonPoint.create(toLat, toLon);
        } else {
            toLat = Math.toDegrees(MapMath.authlat(Math.asin(2.0 * y * this.scaleFactor / this.qp), this.apa));
            toLon = this.lon0 + Math.toDegrees(x / this.scaleFactor);
        }
        return LatLonPoint.create(toLat, toLon);
    }

    @Override
    public boolean crossSeam(ProjectionPoint pt1, ProjectionPoint pt2) {
        return false;
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        CylindricalEqualAreaProjection that = (CylindricalEqualAreaProjection)o;
        if (Double.compare(that.falseEasting, this.falseEasting) != 0) {
            return false;
        }
        if (Double.compare(that.falseNorthing, this.falseNorthing) != 0) {
            return false;
        }
        if (Double.compare(that.projectionLongitude, this.projectionLongitude) != 0) {
            return false;
        }
        if (Double.compare(that.scaleFactor, this.scaleFactor) != 0) {
            return false;
        }
        return Objects.equals(this.earth, that.earth);
    }

    public int hashCode() {
        long temp = this.scaleFactor != 0.0 ? Double.doubleToLongBits(this.scaleFactor) : 0L;
        int result = (int)(temp ^ temp >>> 32);
        temp = this.projectionLongitude != 0.0 ? Double.doubleToLongBits(this.projectionLongitude) : 0L;
        result = 31 * result + (int)(temp ^ temp >>> 32);
        temp = this.falseEasting != 0.0 ? Double.doubleToLongBits(this.falseEasting) : 0L;
        result = 31 * result + (int)(temp ^ temp >>> 32);
        temp = this.falseNorthing != 0.0 ? Double.doubleToLongBits(this.falseNorthing) : 0L;
        result = 31 * result + (int)(temp ^ temp >>> 32);
        result = 31 * result + (this.earth != null ? this.earth.hashCode() : 0);
        return result;
    }
}

