/*
 * Decompiled with CFR 0.152.
 */
package ucar.unidata.geoloc.projection.proj4;

import java.util.Objects;
import javax.annotation.concurrent.Immutable;
import ucar.unidata.geoloc.Earth;
import ucar.unidata.geoloc.LatLonPoint;
import ucar.unidata.geoloc.Projection;
import ucar.unidata.geoloc.ProjectionPoint;
import ucar.unidata.geoloc.projection.AbstractProjection;
import ucar.unidata.geoloc.projection.proj4.MapMath;

@Immutable
public class EquidistantAzimuthalProjection
extends AbstractProjection {
    private static final int NORTH_POLE = 1;
    private static final int SOUTH_POLE = 2;
    private static final int EQUATOR = 3;
    private static final int OBLIQUE = 4;
    private static final double TOL = 1.0E-8;
    private final double lat0;
    private final double lon0;
    private final double projectionLatitude;
    private final double projectionLongitude;
    private final double falseEasting;
    private final double falseNorthing;
    private final Earth earth;
    private final double e;
    private final double es;
    private final double one_es;
    private final double totalScale;
    private final int mode;
    private final double sinphi0;
    private final double cosphi0;
    private final double[] en;
    private final double N1;
    private final double Mp;
    private final double He;
    private final double G;

    public EquidistantAzimuthalProjection() {
        this(90.0, 0.0, 0.0, 0.0, new Earth());
    }

    public EquidistantAzimuthalProjection(double lat0, double lon0, double falseEasting, double falseNorthing, Earth earth) {
        super("EquidistantAzimuthalProjection", false);
        Objects.requireNonNull(earth, "Azimuthal equidistant constructor requires non-null Earth");
        this.lat0 = lat0;
        this.lon0 = lon0;
        this.projectionLatitude = Math.toRadians(lat0);
        this.projectionLongitude = Math.toRadians(lon0);
        this.falseEasting = falseEasting;
        this.falseNorthing = falseNorthing;
        this.earth = earth;
        this.e = earth.getEccentricity();
        this.es = earth.getEccentricitySquared();
        this.one_es = 1.0 - this.es;
        this.totalScale = earth.getMajor() * 0.001;
        this.addParameter("grid_mapping_name", "azimuthal_equidistant");
        this.addParameter("latitude_of_projection_origin", lat0);
        this.addParameter("longitude_of_central_meridian", lon0);
        if (falseEasting != 0.0 || falseNorthing != 0.0) {
            this.addParameter("false_easting", falseEasting);
            this.addParameter("false_northing", falseNorthing);
            this.addParameter("units", "km");
        }
        this.addParameter("semi_major_axis", earth.getMajor());
        this.addParameter("inverse_flattening", 1.0 / earth.getFlattening());
        if (Math.abs(Math.abs(this.projectionLatitude) - 1.5707963267948966) < 1.0E-10) {
            this.mode = this.projectionLatitude < 0.0 ? 2 : 1;
            this.sinphi0 = this.projectionLatitude < 0.0 ? -1.0 : 1.0;
            this.cosphi0 = 0.0;
        } else if (Math.abs(this.projectionLatitude) < 1.0E-10) {
            this.mode = 3;
            this.sinphi0 = 0.0;
            this.cosphi0 = 1.0;
        } else {
            this.mode = 4;
            this.sinphi0 = Math.sin(this.projectionLatitude);
            this.cosphi0 = Math.cos(this.projectionLatitude);
        }
        double[] enTemp = null;
        double N1Temp = Double.NaN;
        double MpTemp = Double.NaN;
        double HeTemp = Double.NaN;
        double GTemp = Double.NaN;
        if (!earth.isSpherical()) {
            enTemp = MapMath.enfn(this.es);
            switch (this.mode) {
                case 1: {
                    MpTemp = MapMath.mlfn(1.5707963267948966, 1.0, 0.0, enTemp);
                    break;
                }
                case 2: {
                    MpTemp = MapMath.mlfn(-1.5707963267948966, -1.0, 0.0, enTemp);
                    break;
                }
                case 3: 
                case 4: {
                    N1Temp = 1.0 / Math.sqrt(1.0 - this.es * this.sinphi0 * this.sinphi0);
                    HeTemp = this.e / Math.sqrt(this.one_es);
                    GTemp = this.sinphi0 * HeTemp;
                    HeTemp *= this.cosphi0;
                }
            }
        }
        this.en = enTemp;
        this.N1 = N1Temp;
        this.Mp = MpTemp;
        this.He = HeTemp;
        this.G = GTemp;
    }

    @Override
    public Projection constructCopy() {
        return new EquidistantAzimuthalProjection(this.lat0, this.lon0, this.falseEasting, this.falseNorthing, this.earth);
    }

    @Override
    public ProjectionPoint latLonToProj(LatLonPoint latlon) {
        double lam = Math.toRadians(latlon.getLongitude() - this.lon0);
        double phi = Math.toRadians(latlon.getLatitude());
        double toX = 0.0;
        double toY = 0.0;
        if (this.earth.isSpherical()) {
            double sinphi = Math.sin(phi);
            double cosphi = Math.cos(phi);
            double coslam = Math.cos(lam);
            switch (this.mode) {
                case 3: 
                case 4: {
                    toY = this.mode == 3 ? cosphi * coslam : this.sinphi0 * sinphi + this.cosphi0 * cosphi * coslam;
                    if (Math.abs(Math.abs(toY) - 1.0) < 1.0E-8) {
                        if (toY < 0.0) {
                            throw new IllegalStateException();
                        }
                        toX = 0.0;
                        toY = 0.0;
                        break;
                    }
                    double y = Math.acos(toY);
                    y /= Math.sin(y);
                    double x = y * cosphi * Math.sin(lam);
                    toY = y *= this.mode == 3 ? sinphi : this.cosphi0 * sinphi - this.sinphi0 * cosphi * coslam;
                    toX = x;
                    break;
                }
                case 1: {
                    phi = -phi;
                    coslam = -coslam;
                }
                case 2: {
                    if (Math.abs(phi - 1.5707963267948966) < 1.0E-10) {
                        throw new IllegalStateException();
                    }
                    double y = 1.5707963267948966 + phi;
                    double x = y * Math.sin(lam);
                    toY = y *= coslam;
                    toX = x;
                }
            }
        } else {
            double coslam = Math.cos(lam);
            double cosphi = Math.cos(phi);
            double sinphi = Math.sin(phi);
            switch (this.mode) {
                case 1: {
                    coslam = -coslam;
                }
                case 2: {
                    double rho = Math.abs(this.Mp - MapMath.mlfn(phi, sinphi, cosphi, this.en));
                    double x = rho * Math.sin(lam);
                    toY = rho * coslam;
                    toX = x;
                    break;
                }
                case 3: 
                case 4: {
                    if (Math.abs(lam) < 1.0E-10 && Math.abs(phi - this.projectionLatitude) < 1.0E-10) {
                        toY = 0.0;
                        toX = 0.0;
                        break;
                    }
                    double t = Math.atan2(this.one_es * sinphi + this.es * this.N1 * this.sinphi0 * Math.sqrt(1.0 - this.es * sinphi * sinphi), cosphi);
                    double ct = Math.cos(t);
                    double st = Math.sin(t);
                    double Az = Math.atan2(Math.sin(lam) * ct, this.cosphi0 * st - this.sinphi0 * coslam * ct);
                    double cA = Math.cos(Az);
                    double sA = Math.sin(Az);
                    double s = MapMath.asin(Math.abs(sA) < 1.0E-8 ? (this.cosphi0 * st - this.sinphi0 * coslam * ct) / cA : Math.sin(lam) * ct / sA);
                    double H = this.He * cA;
                    double H2 = H * H;
                    double c = this.N1 * s * (1.0 + s * s * (-H2 * (1.0 - H2) / 6.0 + s * (this.G * H * (1.0 - 2.0 * H2 * H2) / 8.0 + s * ((H2 * (4.0 - 7.0 * H2) - 3.0 * this.G * this.G * (1.0 - 7.0 * H2)) / 120.0 - s * this.G * H / 48.0))));
                    toY = c * cA;
                    toX = c * sA;
                }
            }
        }
        return ProjectionPoint.create(this.totalScale * toX + this.falseEasting, this.totalScale * toY + this.falseNorthing);
    }

    @Override
    public LatLonPoint projToLatLon(ProjectionPoint ppt) {
        double toLon;
        double toLat;
        double x = (ppt.getX() - this.falseEasting) / this.totalScale;
        double y = (ppt.getY() - this.falseNorthing) / this.totalScale;
        if (this.earth.isSpherical()) {
            double d;
            double c_rh = MapMath.distance(x, y);
            if (d > Math.PI) {
                if (c_rh - 1.0E-10 > Math.PI) {
                    throw new IllegalStateException();
                }
                c_rh = Math.PI;
            } else if (c_rh < 1.0E-10) {
                return LatLonPoint.create(this.lat0, 0.0);
            }
            if (this.mode == 4 || this.mode == 3) {
                double sinc = Math.sin(c_rh);
                double cosc = Math.cos(c_rh);
                if (this.mode == 3) {
                    toLat = Math.toDegrees(MapMath.asin(y * sinc / c_rh));
                    x *= sinc;
                    y = cosc * c_rh;
                } else {
                    toLat = Math.toDegrees(MapMath.asin(cosc * this.sinphi0 + y * sinc * this.cosphi0 / c_rh));
                    y = (cosc - this.sinphi0 * MapMath.sind(toLat)) * c_rh;
                    x *= sinc * this.cosphi0;
                }
                toLon = Math.toDegrees(y == 0.0 ? 0.0 : Math.atan2(x, y));
            } else if (this.mode == 1) {
                toLat = Math.toDegrees(1.5707963267948966 - c_rh);
                toLon = Math.toDegrees(Math.atan2(x, -y));
            } else {
                toLat = Math.toDegrees(c_rh - 1.5707963267948966);
                toLon = Math.toDegrees(Math.atan2(x, y));
            }
        } else {
            double d;
            double c = MapMath.distance(x, y);
            if (d < 1.0E-10) {
                return LatLonPoint.create(this.lat0, 0.0);
            }
            if (this.mode == 4 || this.mode == 3) {
                double d2;
                double Az = Math.atan2(x, y);
                double cosAz = Math.cos(Az);
                double t = this.cosphi0 * cosAz;
                double B = this.es * t / this.one_es;
                double A = -B * t;
                double D = c / this.N1;
                double E = D * (1.0 - D * D * (A * (1.0 + A) / 6.0 + (B *= 3.0 * (1.0 - A) * this.sinphi0) * (1.0 + 3.0 * A) * D / 24.0));
                double F = 1.0 - E * E * (A / 2.0 + B * E / 6.0);
                double psi = MapMath.asin(this.sinphi0 * Math.cos(E) + t * Math.sin(E));
                toLon = Math.toDegrees(MapMath.asin(Math.sin(Az) * Math.sin(E) / Math.cos(psi)));
                t = Math.abs(psi);
                toLat = d2 < 1.0E-10 ? 0.0 : (Math.abs(t - 1.5707963267948966) < 0.0 ? Math.toDegrees(1.5707963267948966) : Math.toDegrees(Math.atan((1.0 - this.es * F * this.sinphi0 / Math.sin(psi)) * Math.tan(psi) / this.one_es)));
            } else {
                toLat = Math.toDegrees(MapMath.inv_mlfn(this.mode == 1 ? this.Mp - c : this.Mp + c, this.es, this.en));
                toLon = Math.toDegrees(Math.atan2(x, this.mode == 1 ? -y : y));
            }
        }
        return LatLonPoint.create(toLat, toLon + this.lon0);
    }

    @Override
    public String paramsToString() {
        return null;
    }

    @Override
    public boolean crossSeam(ProjectionPoint pt1, ProjectionPoint pt2) {
        return false;
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        EquidistantAzimuthalProjection that = (EquidistantAzimuthalProjection)o;
        if (Double.compare(that.falseEasting, this.falseEasting) != 0) {
            return false;
        }
        if (Double.compare(that.falseNorthing, this.falseNorthing) != 0) {
            return false;
        }
        if (Double.compare(that.projectionLatitude, this.projectionLatitude) != 0) {
            return false;
        }
        if (Double.compare(that.projectionLongitude, this.projectionLongitude) != 0) {
            return false;
        }
        return Objects.equals(this.earth, that.earth);
    }

    public int hashCode() {
        long temp = this.projectionLatitude != 0.0 ? Double.doubleToLongBits(this.projectionLatitude) : 0L;
        int result = (int)(temp ^ temp >>> 32);
        temp = this.projectionLongitude != 0.0 ? Double.doubleToLongBits(this.projectionLongitude) : 0L;
        result = 31 * result + (int)(temp ^ temp >>> 32);
        temp = this.falseEasting != 0.0 ? Double.doubleToLongBits(this.falseEasting) : 0L;
        result = 31 * result + (int)(temp ^ temp >>> 32);
        temp = this.falseNorthing != 0.0 ? Double.doubleToLongBits(this.falseNorthing) : 0L;
        result = 31 * result + (int)(temp ^ temp >>> 32);
        result = 31 * result + (this.earth != null ? this.earth.hashCode() : 0);
        return result;
    }
}

