/*
 * Decompiled with CFR 0.152.
 */
package ucar.unidata.io.s3;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.Objects;
import java.util.Optional;
import javax.annotation.Nullable;
import javax.annotation.concurrent.Immutable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Immutable
public final class CdmS3Uri {
    private static final Logger logger = LoggerFactory.getLogger(CdmS3Uri.class);
    private static final String SCHEME_HTTP = "http";
    private static final String SCHEME_HTTPS = "https";
    public static final String SCHEME_CDM_S3 = "cdms3";
    private static final String SCHEME_CDM_S3_DEPRECATED = "s3";
    private static final String DELIMITER = "delimiter";
    private final String bucket;
    private final String key;
    private final String profile;
    private final String uriString;
    private final URI endpoint;
    private final String delimiter;
    private final String fragment;

    public CdmS3Uri(String cdmUriString) throws URISyntaxException {
        URI cdmS3Uri = new URI(cdmUriString);
        String scheme = cdmS3Uri.getScheme();
        if (scheme == null) {
            throw new URISyntaxException(cdmS3Uri.toString(), String.format("A CdmS3Uri must have a defined scheme (i.e. start with %s).", SCHEME_CDM_S3));
        }
        if (scheme.equalsIgnoreCase(SCHEME_CDM_S3_DEPRECATED)) {
            logger.warn("Use of the s3 scheme is deprecated. Please switch to the cdms3 scheme.");
            String path = cdmS3Uri.getPath();
            path = path == null ? "" : "?" + path.substring(1);
            String fragment = cdmS3Uri.getFragment();
            fragment = fragment == null ? "" : "#" + fragment;
            path = path + fragment;
            String updatedUri = String.format("cdms3:%s%s", cdmS3Uri.getHost(), path);
            logger.warn(String.format("Using updated URI: %s", updatedUri));
            cdmS3Uri = new URI(updatedUri);
            scheme = cdmS3Uri.getScheme();
        }
        if (!scheme.equalsIgnoreCase(SCHEME_CDM_S3)) {
            throw new URISyntaxException(cdmS3Uri.toString(), String.format("A CdmS3Uri must use the %s scheme.", SCHEME_CDM_S3));
        }
        this.bucket = this.getBucketName(cdmS3Uri);
        this.key = this.getObjectKey(cdmS3Uri);
        this.profile = this.getProfile(cdmS3Uri);
        this.uriString = cdmS3Uri.toString();
        this.endpoint = this.getEndpoint(cdmS3Uri, this.bucket);
        this.delimiter = this.getKeyDelimiter(cdmS3Uri);
        this.fragment = cdmS3Uri.getFragment();
    }

    public String getBucket() {
        return this.bucket;
    }

    public Optional<String> getKey() {
        return Optional.ofNullable(this.key);
    }

    public Optional<String> getProfile() {
        return Optional.ofNullable(this.profile);
    }

    public Optional<URI> getEndpoint() {
        return Optional.ofNullable(this.endpoint);
    }

    public Optional<String> getDelimiter() {
        return Optional.ofNullable(this.delimiter);
    }

    public String toString() {
        return this.uriString;
    }

    public CdmS3Uri resolveNewKey(String newKey) throws URISyntaxException {
        String newUri = this.uriString;
        if (this.key != null) {
            newUri = newUri.replace(this.key, newKey);
        } else {
            int fragLoc = newUri.lastIndexOf("#");
            if (fragLoc > 0) {
                String fragment = newUri.substring(fragLoc);
                newUri = newUri.substring(0, fragLoc);
                newUri = newUri + "?" + newKey + fragment;
            }
        }
        return new CdmS3Uri(newUri);
    }

    public boolean isAws() {
        boolean isAws = this.endpoint != null ? this.endpoint.getHost().contains("amazonaws.com") : true;
        return isAws;
    }

    @Nullable
    private String getProfile(URI cdmUri) {
        String profile = null;
        if (cdmUri.getAuthority() != null) {
            profile = cdmUri.getRawUserInfo();
        }
        return profile;
    }

    @Nullable
    private URI getEndpoint(URI cdmUri, String bucketName) {
        String host;
        URI s3Endpoint = null;
        if (cdmUri.getAuthority() != null && !(host = cdmUri.getHost()).equalsIgnoreCase("aws")) {
            String endpointPath;
            int endOfPath;
            String scheme;
            boolean keepPort = true;
            int port = cdmUri.getPort();
            switch (port) {
                case -1: 
                case 443: {
                    keepPort = false;
                    scheme = SCHEME_HTTPS;
                    break;
                }
                case 80: {
                    keepPort = false;
                    scheme = SCHEME_HTTP;
                    break;
                }
                case 7001: 
                case 8008: 
                case 8080: 
                case 9080: 
                case 16080: {
                    scheme = SCHEME_HTTP;
                    break;
                }
                default: {
                    scheme = SCHEME_HTTPS;
                }
            }
            if (keepPort) {
                host = host + ":" + port;
            }
            if ((endOfPath = (endpointPath = cdmUri.getPath()).lastIndexOf(bucketName)) > 0) {
                endpointPath = endpointPath.substring(0, endOfPath);
            }
            s3Endpoint = URI.create(scheme + "://" + host + endpointPath);
        }
        return s3Endpoint;
    }

    private String getBucketName(URI cdmUri) {
        String bucketName;
        if (cdmUri.getAuthority() != null) {
            String path = cdmUri.getPath();
            path = path.endsWith("/") ? path.substring(0, path.length() - 1) : path;
            int bucketIndex = path.lastIndexOf(47);
            bucketName = bucketIndex >= 0 ? path.substring(bucketIndex + 1) : path;
        } else {
            String schemeSpecificPart = cdmUri.getSchemeSpecificPart();
            int bucketEndIndex = schemeSpecificPart.indexOf(63);
            bucketName = bucketEndIndex >= 0 ? schemeSpecificPart.substring(0, bucketEndIndex) : schemeSpecificPart;
        }
        return bucketName;
    }

    @Nullable
    private String getObjectKey(URI cdmUri) {
        String key = null;
        if (cdmUri.getAuthority() != null) {
            key = cdmUri.getQuery();
        } else {
            String schemeSpecificPart = cdmUri.getSchemeSpecificPart();
            int bucketEndIndex = schemeSpecificPart.indexOf(63);
            int queryStartIndex = bucketEndIndex + 1;
            if (bucketEndIndex >= 0 && queryStartIndex < schemeSpecificPart.length()) {
                key = schemeSpecificPart.substring(queryStartIndex);
            }
        }
        return key;
    }

    private String getKeyDelimiter(URI cdmUri) {
        String delimiter = null;
        String config = cdmUri.getFragment();
        if (config != null) {
            for (String kvp : config.split("&")) {
                String[] splitKvp = kvp.split("=");
                if (splitKvp[0].equalsIgnoreCase(DELIMITER) && splitKvp.length == 2) {
                    delimiter = kvp.split("=")[1];
                    continue;
                }
                logger.debug("Unknown configuration option encountered: {}", (Object)kvp);
            }
        }
        return delimiter;
    }

    public int hashCode() {
        return Objects.hash(this.bucket, this.key, this.profile, this.uriString, this.endpoint, this.delimiter, this.fragment);
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof CdmS3Uri)) {
            return false;
        }
        CdmS3Uri cdmS3Uri = (CdmS3Uri)o;
        return this.bucket.equals(cdmS3Uri.bucket) && this.uriString.equals(cdmS3Uri.uriString) && Objects.equals(this.key, cdmS3Uri.key) && Objects.equals(this.profile, cdmS3Uri.profile) && Objects.equals(this.endpoint, cdmS3Uri.endpoint) && Objects.equals(this.delimiter, cdmS3Uri.delimiter) && Objects.equals(this.fragment, cdmS3Uri.fragment);
    }
}

