/*
 * Copyright 1998-2014 University Corporation for Atmospheric Research/Unidata
 *
 *   Portions of this software were developed by the Unidata Program at the
 *   University Corporation for Atmospheric Research.
 *
 *   Access and use of this software shall impose the following obligations
 *   and understandings on the user. The user is granted the right, without
 *   any fee or cost, to use, copy, modify, alter, enhance and distribute
 *   this software, and any derivative works thereof, and its supporting
 *   documentation for any purpose whatsoever, provided that this entire
 *   notice appears in all copies of the software, derivative works and
 *   supporting documentation.  Further, UCAR requests that the user credit
 *   UCAR/Unidata in any publications that result from the use of this
 *   software or in any product that includes this software. The names UCAR
 *   and/or Unidata, however, may not be used in any advertising or publicity
 *   to endorse or promote any products or commercial entity unless specific
 *   written permission is obtained from UCAR/Unidata. The user also
 *   understands that UCAR/Unidata is not obligated to provide the user with
 *   any support, consulting, training or assistance of any kind with regard
 *   to the use, operation and performance of this software nor to provide
 *   the user with any updates, revisions, new versions or "bug fixes."
 *
 *   THIS SOFTWARE IS PROVIDED BY UCAR/UNIDATA "AS IS" AND ANY EXPRESS OR
 *   IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 *   WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 *   DISCLAIMED. IN NO EVENT SHALL UCAR/UNIDATA BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE ACCESS, USE OR PERFORMANCE OF THIS SOFTWARE.
 */
package ucar.ma2;

import java.io.Closeable;
import java.io.IOException;

/**
 * An iterator over StructureData.
  * Make sure that you call close().
  * Best to put in a try/finally block like:
<pre>
try {
  while (iter.hasNext())
   process(iter.next());
 } finally {
   iter.finish();
 }
</pre>
 * @author caron
 * @since Feb 23, 2008
 */
public interface StructureDataIterator extends Closeable {

  /**
   * See if theres more StructureData in the iteration.
   * You must always call this before calling next().
   *
   * @return true if more records are available
   * @throws java.io.IOException on read error
   */
  boolean hasNext() throws IOException;

  /**
   * Get the next StructureData in the iteration.
   *
   * @return next StructureData record.
   * @throws java.io.IOException on read error
   */
  StructureData next() throws IOException;

  /**
   * Hint to use this much memory in buffering the iteration.
   * No guarentee that it will be used by the implementation.
   *
   * @param bytes amount of memory in bytes
   */
  default void setBufferSize(int bytes)  {
      // doan do nuthin
    }

  /**
   * Start the iteration over again.
   *
   * @return a new or reset iterator.
   */
  StructureDataIterator reset();

  int getCurrentRecno();

  /**
   * Make sure that the iterator is complete, and recover resources.
   * Best to put in a try/finally block like:
   * <pre>
   try (StructureDataIterator iter = obj.getStructureDataIterator()) {
     while (iter.hasNext())
      process(iter.next());
    }
   </pre>
   */
  default void close() {
    // doan do nuthin
  }

  ////////////

  /**
   * @deprecated use close() or try-with-resource
   */
  default void finish() {
    close();
  }




}
