/*
 * Decompiled with CFR 0.152.
 */
package dap4.core.util;

import dap4.core.dmr.AtomicType;
import dap4.core.dmr.DapEnum;
import dap4.core.dmr.DapType;
import dap4.core.util.ConversionException;
import dap4.core.util.DapException;
import dap4.core.util.DapUtil;
import dap4.core.util.Escape;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.net.MalformedURLException;
import java.net.URL;

public abstract class Convert {
    static final long INT8_MAX = 127L;
    static final long INT8_MIN = -128L;
    static final long UINT8_MAX = 255L;
    static final long UINT8_MIN = 0L;
    static final long INT16_MAX = 32767L;
    static final long INT16_MIN = -32768L;
    static final long UINT16_MAX = 65535L;
    static final long UINT16_MIN = 0L;
    static final long INT32_MAX = Integer.MAX_VALUE;
    static final long INT32_MIN = Integer.MIN_VALUE;
    static final long UINT32_MAX = 0xFFFFFFFFL;
    static final long UINT32_MIN = 0L;
    static final long INT64_MAX = Long.MAX_VALUE;
    static final long INT64_MIN = Long.MIN_VALUE;
    static final long UINT64_MIN = 0L;
    static final BigInteger BIG_INT64_MAX = BigInteger.valueOf(Long.MAX_VALUE);
    static final BigInteger BIG_INT64_MIN = BigInteger.valueOf(Long.MIN_VALUE);
    static final BigInteger BIG_UINT64_MAX = new BigInteger("FFFFFFFFFFFFFFFF", 16);
    static final BigInteger BIG_UINT64_MIN = BigInteger.ZERO;

    public static DapType upcastType(DapType basetype) {
        switch (basetype.getAtomicType()) {
            case UInt8: {
                return DapType.INT16;
            }
            case UInt16: {
                return DapType.INT32;
            }
            case UInt32: {
                return DapType.INT64;
            }
        }
        return basetype;
    }

    public static Object upcast(Object o, DapType srctype) {
        Object result = null;
        AtomicType otype = AtomicType.classToType(o);
        AtomicType srcatomtype = srctype.getAtomicType();
        if (otype == null) {
            throw new ConversionException("Unexpected value type: " + o.getClass());
        }
        switch (srcatomtype) {
            case UInt8: {
                long lvalue = ((Byte)o).longValue();
                result = new Short((short)(lvalue &= 0xFFL));
                break;
            }
            case UInt16: {
                long lvalue = ((Short)o).longValue();
                result = new Integer((int)(lvalue &= 0xFFFFL));
                break;
            }
            case UInt32: {
                long lvalue = ((Integer)o).longValue();
                result = new Long(lvalue &= 0xFFFFFFFFL);
                break;
            }
            case Structure: {
                throw new ConversionException("Cannot convert struct");
            }
            default: {
                result = o;
            }
        }
        return result;
    }

    public static int getJavaSize(DapType daptype) {
        AtomicType atomictype = daptype.getPrimitiveType();
        return Convert.getJavaSize(atomictype);
    }

    public static int getJavaSize(AtomicType atomtype) {
        switch (atomtype) {
            case UInt8: 
            case Char: 
            case Int8: {
                return 1;
            }
            case UInt16: 
            case Int16: {
                return 2;
            }
            case UInt32: 
            case Int32: {
                return 4;
            }
            case Int64: 
            case UInt64: {
                return 8;
            }
            case Float32: {
                return 4;
            }
            case Float64: {
                return 8;
            }
        }
        return 0;
    }

    static long minmax(long value, long min, long max) {
        if (value < min) {
            return min;
        }
        if (value > max) {
            return max;
        }
        return value;
    }

    public static Object convert(DapType dsttype, DapType srctype, Object value) {
        Object result = null;
        if (dsttype == srctype) {
            return value;
        }
        AtomicType srcatomtype = srctype.getAtomicType();
        AtomicType dstatomtype = dsttype.getAtomicType();
        if (srcatomtype.isEnumType()) {
            return Convert.convert(dsttype, ((DapEnum)srctype).getBaseType(), value);
        }
        assert (!srcatomtype.isEnumType());
        long lvalue = 0L;
        double dvalue = 0.0;
        if (srcatomtype.isNumericType()) {
            lvalue = Convert.longValue(srctype, value);
        } else if (srcatomtype.isFloatType()) {
            dvalue = Convert.doubleValue(srctype, value);
        }
        boolean fail = false;
        switch (dstatomtype) {
            case Char: {
                if (!srcatomtype.isNumericType()) {
                    fail = true;
                    break;
                }
                result = Character.valueOf((char)(lvalue &= 0xFFL));
                break;
            }
            case UInt8: {
                if (!srcatomtype.isNumericType()) {
                    fail = true;
                    break;
                }
                result = (byte)(lvalue &= 0xFFL);
                break;
            }
            case Int8: {
                if (!srcatomtype.isNumericType()) {
                    fail = true;
                    break;
                }
                result = (byte)lvalue;
                break;
            }
            case UInt16: {
                if (!srcatomtype.isNumericType()) {
                    fail = true;
                    break;
                }
                result = (short)(lvalue &= 0xFFFFL);
                break;
            }
            case Int16: {
                if (!srcatomtype.isNumericType()) {
                    fail = true;
                    break;
                }
                result = (short)lvalue;
                break;
            }
            case UInt32: {
                if (!srcatomtype.isNumericType()) {
                    fail = true;
                    break;
                }
                result = (int)(lvalue &= 0xFFFFL);
                break;
            }
            case Int32: {
                if (!srcatomtype.isNumericType()) {
                    fail = true;
                    break;
                }
                result = (int)lvalue;
                break;
            }
            case Int64: 
            case UInt64: {
                if (!srcatomtype.isNumericType()) {
                    fail = true;
                    break;
                }
                result = lvalue;
                break;
            }
            case Float32: {
                if (!srcatomtype.isNumericType()) {
                    fail = true;
                    break;
                }
                result = Float.valueOf((float)dvalue);
                break;
            }
            case Float64: {
                if (!srcatomtype.isNumericType()) {
                    fail = true;
                    break;
                }
                result = dvalue;
                break;
            }
            case String: 
            case URL: {
                if (srcatomtype == AtomicType.Opaque) break;
                result = value.toString();
                break;
            }
            case Opaque: {
                if (srcatomtype != AtomicType.Opaque) {
                    fail = true;
                    break;
                }
                result = value;
                break;
            }
            case Enum: {
                if (!srcatomtype.isIntegerType()) {
                    fail = true;
                    break;
                }
                DapEnum en = (DapEnum)dsttype;
                if (en.lookup(lvalue) == null) {
                    throw new ConversionException("Enum constant failure");
                }
                result = lvalue;
                break;
            }
            default: {
                fail = true;
            }
        }
        if (fail) {
            throw new ConversionException(String.format("Cannot convert: %s -> %s", new Object[]{srcatomtype, dstatomtype}));
        }
        return result;
    }

    public static long longValue(DapType srctype, Object value) {
        AtomicType srcatomtype = srctype.getAtomicType();
        if (srcatomtype.isEnumType()) {
            return Convert.longValue(((DapEnum)srctype).getBaseType(), value);
        }
        assert (!srctype.isEnumType());
        if (srcatomtype.isCharType()) {
            return (long)((Character)value).charValue() & 0xFFL;
        }
        if (srcatomtype.isIntegerType()) {
            return ((Number)value).longValue();
        }
        if (srcatomtype == AtomicType.Float32) {
            return ((Float)value).longValue();
        }
        if (srcatomtype == AtomicType.Float64) {
            return ((Double)value).longValue();
        }
        throw new ConversionException(String.format("Cannot convert: %s -> long", new Object[]{srcatomtype}));
    }

    public static double doubleValue(DapType srctype, Object value) {
        AtomicType srcatomtype = srctype.getAtomicType();
        if (srcatomtype.isEnumType()) {
            return Convert.doubleValue(((DapEnum)srctype).getBaseType(), value);
        }
        assert (!srcatomtype.isEnumType());
        double dvalue = 0.0;
        if (srcatomtype == AtomicType.UInt64) {
            BigInteger bi = Convert.toBigInteger((Long)value);
            BigDecimal bd = new BigDecimal(bi);
            dvalue = bd.doubleValue();
        } else if (srcatomtype.isIntegerType() || srcatomtype.isCharType()) {
            long lvalue = Convert.longValue(srctype, value);
            dvalue = lvalue;
        } else if (srcatomtype == AtomicType.Float32) {
            float f = ((Float)value).floatValue();
            dvalue = f;
        } else if (srcatomtype == AtomicType.Float64) {
            dvalue = (Double)value;
        } else {
            throw new ConversionException(String.format("Cannot convert: %s -> double", new Object[]{srcatomtype}));
        }
        return dvalue;
    }

    public static BigInteger toBigInteger(long l) {
        BigInteger bi = BigInteger.valueOf(l);
        bi = bi.and(DapUtil.BIG_UMASK64);
        return bi;
    }

    public static long forceRange(AtomicType basetype, long value) {
        assert (basetype.isIntegerType()) : "Internal error";
        switch (basetype) {
            case Char: {
                value = Convert.minmax(value, 0L, 255L);
                break;
            }
            case Int8: {
                value = Convert.minmax(value, -128L, 127L);
                break;
            }
            case UInt8: {
                value &= 0xFFL;
                break;
            }
            case Int16: {
                value = Convert.minmax(value, -32768L, 32767L);
                break;
            }
            case UInt16: {
                value &= 0xFFFFL;
                break;
            }
            case Int32: {
                value = Convert.minmax(value, Integer.MIN_VALUE, Integer.MAX_VALUE);
                break;
            }
            case UInt32: {
                value &= 0xFFFFFFFFL;
                break;
            }
            case Int64: 
            case UInt64: {
                break;
            }
        }
        return value;
    }

    public static double forceRange(AtomicType basetype, double value) throws DapException {
        assert (basetype.isFloatType()) : "Internal error";
        if (basetype == AtomicType.Float32) {
            float fvalue = (float)value;
            value = fvalue;
        }
        return value;
    }

    public static Object fromString(String value, DapType dsttype) {
        if (value == null) {
            return value;
        }
        assert (dsttype != null);
        AtomicType atomtype = dsttype.getAtomicType();
        long lvalue = 0L;
        if (atomtype.isIntegerType() || atomtype.isCharType()) {
            BigInteger bi = BigInteger.ZERO;
            try {
                bi = new BigInteger(value);
                lvalue = bi.longValue();
            }
            catch (NumberFormatException nfe) {
                throw new ConversionException("Expected integer value: " + value);
            }
            lvalue = Convert.forceRange(atomtype, lvalue);
            switch (atomtype) {
                case UInt8: 
                case Int8: {
                    return (byte)lvalue;
                }
                case UInt16: 
                case Int16: {
                    return (short)lvalue;
                }
                case Int64: 
                case UInt64: {
                    return lvalue;
                }
                case Char: {
                    return Character.valueOf((char)lvalue);
                }
            }
            assert (false);
        } else if (atomtype.isFloatType()) {
            double d = 0.0;
            try {
                d = Double.parseDouble(value);
            }
            catch (NumberFormatException nfe) {
                throw new ConversionException("Expected float value: " + value);
            }
            switch (atomtype) {
                case Float32: {
                    return new Float(d);
                }
                case Float64: {
                    return new Double(d);
                }
            }
            assert (false);
        } else {
            if (atomtype.isStringType()) {
                if (atomtype == AtomicType.URL) {
                    value = value.trim();
                    URL url = null;
                    try {
                        url = new URL(value);
                    }
                    catch (MalformedURLException mue) {
                        throw new ConversionException("Illegal URL value: " + value);
                    }
                }
                return value.toString();
            }
            if (atomtype.isOpaqueType()) {
                String opaque = value.trim();
                if (!opaque.startsWith("0x") && !opaque.startsWith("0X")) {
                    throw new ConversionException("Illegal opaque value: " + value);
                }
                int len = (opaque = opaque.substring(2, opaque.length())).length();
                if (len % 2 == 1) {
                    ++len;
                    opaque = opaque + '0';
                }
                byte[] b = new byte[len];
                int index = 0;
                for (int i = 0; i < len; i += 2) {
                    int digit1 = Escape.fromHex(opaque.charAt(i));
                    int digit2 = Escape.fromHex(opaque.charAt(i + 1));
                    if (digit1 < 0 || digit2 < 0) {
                        throw new ConversionException("Illegal attribute value: " + value);
                    }
                    b[index++] = (byte)(digit1 << 4 | digit2);
                }
                return b;
            }
            if (atomtype.isEnumType()) {
                String name = value.trim();
                Long Lvalue = ((DapEnum)dsttype).lookup(name);
                if (Lvalue == null) {
                    throw new ConversionException("Illegal enum constant value: " + name);
                }
                return Lvalue;
            }
        }
        throw new ConversionException("Internal error");
    }

    public static String toString(Object value, DapType srctype) {
        StringBuilder buf = new StringBuilder();
        boolean fail = false;
        long lvalue = 0L;
        assert (srctype != null) : "Internal error";
        AtomicType srcatomtype = srctype.getAtomicType();
        boolean charornum = true;
        if (srcatomtype.isCharType()) {
            lvalue = ((Character)value).charValue();
        } else if (srcatomtype.isNumericType()) {
            lvalue = ((Number)value).longValue();
        } else if (srcatomtype == AtomicType.UInt64) {
            lvalue = ((BigInteger)value).longValue();
        } else {
            charornum = false;
        }
        switch (srcatomtype) {
            case UInt8: 
            case Char: {
                if (!charornum) {
                    fail = true;
                    break;
                }
                buf.append("'" + (char)(lvalue &= 0xFFL) + "'");
                break;
            }
            case Int8: {
                if (!charornum) {
                    fail = true;
                    break;
                }
                buf.append(Long.toString(lvalue));
                break;
            }
            case Int16: {
                if (!charornum) {
                    fail = true;
                    break;
                }
                buf.append(Short.toString((short)lvalue));
                break;
            }
            case UInt16: {
                if (!charornum) {
                    fail = true;
                    break;
                }
                buf.append(Long.toString(lvalue & 0xFFFFL));
                break;
            }
            case Int32: {
                if (!charornum) {
                    fail = true;
                    break;
                }
                buf.append(Integer.toString((int)lvalue));
                break;
            }
            case UInt32: {
                if (!charornum) {
                    fail = true;
                    break;
                }
                buf.append(Long.toString(lvalue & 0xFFFFFFFFL));
                break;
            }
            case Int64: {
                if (!charornum) {
                    fail = true;
                    break;
                }
                buf.append(Long.toString(lvalue));
                break;
            }
            case UInt64: {
                if (!charornum) {
                    fail = true;
                    break;
                }
                BigInteger bi = Convert.toBigInteger(lvalue);
                bi = bi.and(DapUtil.BIG_UMASK64);
                buf.append(bi.toString());
                break;
            }
            case Enum: {
                DapEnum en = (DapEnum)srctype;
                String name = en.lookup(lvalue);
                if (name == null) {
                    name = "?";
                }
                buf.append(en.getFQN() + "." + name);
                break;
            }
        }
        if (fail) {
            throw new ConversionException();
        }
        return buf.toString();
    }
}

