/*
 * Copyright (c) 1998-2018 John Caron and University Corporation for Atmospheric Research/Unidata
 * See LICENSE for license information.
 */

package ucar.nc2.grib.collection;

import javax.annotation.Nullable;
import thredds.featurecollection.FeatureCollectionConfig;
import ucar.nc2.time.CalendarDate;
import ucar.unidata.io.RandomAccessFile;
import java.util.List;

/**
 * Read Grib2Partition From ncx Index,
 * Data file never opened.
 *
 * @author John
 * @since 12/7/13
 */
public class Grib2PartitionBuilderFromIndex extends Grib2CollectionBuilderFromIndex {

  // read in the index, index raf already open; return null on failure
  @Nullable
  static Grib2Partition createTimePartitionFromIndex(String name, RandomAccessFile raf, FeatureCollectionConfig config,
      org.slf4j.Logger logger) {

    Grib2PartitionBuilderFromIndex builder = new Grib2PartitionBuilderFromIndex(name, config, logger);
    if (builder.readIndex(raf))
      return new Grib2Partition(builder.pc);

    return null;
  }

  // read in the index, index raf already open; return null on failure
  @Nullable
  static PartitionCollectionMutable openMutablePCFromIndex(String name, RandomAccessFile raf,
      FeatureCollectionConfig config, org.slf4j.Logger logger) {

    Grib2PartitionBuilderFromIndex builder = new Grib2PartitionBuilderFromIndex(name, config, logger);
    if (builder.readIndex(raf))
      return builder.pc;

    return null;
  }

  //////////////////////////////////////////////////////////////////////////////////

  private final PartitionCollectionMutable pc; // build this object

  private Grib2PartitionBuilderFromIndex(String name, FeatureCollectionConfig config, org.slf4j.Logger logger) {
    super(name, config, logger);
    this.pc = new PartitionCollectionMutable(name, null, config, false, logger);
    this.gc = pc;
  }

  @Override
  public String getMagicStart() {
    return Grib2PartitionBuilder.MAGIC_START;
  }

  ///////////////////////////////////////////////////////////////////////////
  // reading ncx

  /*
   * extend GribCollection {
   * repeated Partition partitions = 100;
   * required bool isPartitionOfPartitions = 101;
   * repeated uint32 run2part = 102; // masterRuntime index to partition index
   * }
   */
  @Override
  protected boolean readExtensions(GribCollectionProto.GribCollection proto) {
    pc.isPartitionOfPartitions = proto.getIsPartitionOfPartitions();

    List<Integer> list = proto.getRun2PartList();
    pc.run2part = new int[list.size()];
    int count = 0;
    for (int partno : list)
      pc.run2part[count++] = partno;

    List<ucar.nc2.grib.collection.GribCollectionProto.Partition> partList = proto.getPartitionsList();
    for (ucar.nc2.grib.collection.GribCollectionProto.Partition partProto : partList)
      makePartition(partProto);

    return !partList.isEmpty();
  }

  /*
   * extend Variable {
   * repeated PartitionVariable partition = 100;
   * repeated Parameter vparams = 101; // not used yet
   * }
   */

  @Override
  protected GribCollectionMutable.VariableIndex readVariableExtensions(GribCollectionMutable.GroupGC group,
      GribCollectionProto.Variable proto, GribCollectionMutable.VariableIndex vi) {
    List<GribCollectionProto.PartitionVariable> pvList = proto.getPartVariableList();
    PartitionCollectionMutable.VariableIndexPartitioned vip = pc.makeVariableIndexPartitioned(group, vi, pvList.size());
    vip.setPartitions(pvList);

    // cant put this in the constructor
    vip.ndups = vi.ndups;
    vip.nrecords = vi.nrecords;
    vip.nmissing = vi.nmissing;
    return vip;
  }

  /*
   * message Partition {
   * required string name = 1; // name is used in TDS - eg the subdirectory when generated by TimePartitionCollections
   * required string filename = 2; // the gribCollection.ncx2 file
   * required string directory = 3; // top directory
   * optional uint64 lastModified = 4;
   * }
   */
  private PartitionCollectionMutable.Partition makePartition(GribCollectionProto.Partition proto) {
    long partitionDateMillisecs = proto.getPartitionDate();
    CalendarDate partitionDate = partitionDateMillisecs > 0 ? CalendarDate.of(partitionDateMillisecs) : null;
    return pc.addPartition(proto.getName(), proto.getFilename(), proto.getLastModified(), proto.getLength(), null,
        partitionDate);
  }

}
