/////////////////////////////////////////////////////////////////////////////
// This file is part of the "Java-DAP" project, a Java implementation
// of the OPeNDAP Data Access Protocol.
//
// Copyright (c) 2010, OPeNDAP, Inc.
// Copyright (c) 2002,2003 OPeNDAP, Inc.
//
// Author: James Gallagher <jgallagher@opendap.org>
//
// All rights reserved.
//
// Redistribution and use in source and binary forms,
// with or without modification, are permitted provided
// that the following conditions are met:
//
// - Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
//
// - Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the
// documentation and/or other materials provided with the distribution.
//
// - Neither the name of the OPeNDAP nor the names of its contributors may
// be used to endorse or promote products derived from this software
// without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
// IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
// TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
// PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
// TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
// PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
// LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
// NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
/////////////////////////////////////////////////////////////////////////////


package opendap.dap;

import com.google.common.collect.ImmutableList;
import opendap.dap.parsers.DDSXMLParser;
import java.io.PrintWriter;

/**
 * Contains methods used only by the OPeNDAP constructor classes
 * (<code>DStructure</code>, <code>DSequence</code>, <code>DGrid</code>, and
 * <code>DList</code>).
 *
 * @author jehamby
 * @version $Revision: 19676 $
 * @see DStructure
 * @see DSequence
 * @see DGrid
 */
public abstract class DConstructor extends BaseType {

  /**
   * Constructs a new <code>DConstructor</code>.
   */
  public DConstructor() {
    super();
  }

  /**
   * Constructs a new <code>DConstructor</code> with the given name.
   *
   * @param n The name of the variable.
   */
  public DConstructor(String n) {
    super(n);
  }

  /**
   * Adds a variable to the container.
   *
   * @param v the variable to add.
   * @param part The part of the constructor data to be modified.
   */
  public abstract void addVariable(BaseType v, int part);

  /**
   * Adds a variable to the container. Same as <code>addVariable(v, 0)</code>.
   *
   * @param v the variable to add.
   */
  public final void addVariable(BaseType v) {
    addVariable(v, 0);
  }

  /**
   * Gets the named variable.
   *
   * @param name the name of the variable.
   * @return the named variable.
   * @throws NoSuchVariableException if the named variable does not
   *         exist in this container.
   */
  public abstract BaseType getVariable(String name) throws NoSuchVariableException;

  /**
   * Gets the indexed variable. For a DGrid the index 0 returns the <code>DArray</code> and
   * indexes 1 and higher return the associated map <code>Vector</code>s.
   *
   * @param index the index of the variable in the <code>Vector</code> Vars.
   * @return the named variable.
   * @throws NoSuchVariableException if the named variable does not
   *         exist in this container.
   */
  public abstract BaseType getVar(int index) throws NoSuchVariableException;

  /**
   * Get the number of contained variables (for use with getVar()
   * 
   * @return the number of contained variables
   */
  public abstract int getVarCount();


  /** Return a List of member Variables. */
  public abstract ImmutableList<BaseType> getVariables();

  /**
   * @param bt The BaseType object to search.
   * @return true if some child of the passed BaseType has attributes
   */
  protected boolean someChildHasAttributes(BaseType bt) {
    boolean foundit = false;
    if (bt.hasAttributes())
      return (true);
    if (bt instanceof DConstructor) {
      for (BaseType member : ((DConstructor) bt).getVariables()) {
        foundit = foundit || someChildHasAttributes(member);
      }
    }
    return foundit;
  }

  /**
   *
   * @param pw Where to print
   * @param pad Padding for iondentation (makes the output easier for humans
   *        to read).
   * @param constrained If true then only projected variables (and their
   *        Attributes) will be printed.
   */
  // Coverity[CALL_SUPER]
  public void printXML(PrintWriter pw, String pad, boolean constrained) {

    boolean hasAttributes = hasAttributes();
    boolean hasVariables = !getVariables().isEmpty();

    pw.print(pad + "<" + getTypeName());
    if (getEncodedName() != null) {
      pw.print(" name=\"" + DDSXMLParser.normalizeToXML(getClearName()) + "\"");
    }

    if (hasAttributes || hasVariables) {
      pw.println(">");
      for (String attName : getAttributeNames()) {
        Attribute a = getAttribute(attName);
        if (a != null)
          a.printXML(pw, pad + "\t", constrained);
      }

      for (BaseType bt : getVariables()) {
        bt.printXML(pw, pad + "\t", constrained);
      }
      pw.println(pad + "</" + getTypeName() + ">");

    } else {
      pw.println("/>");
    }

  }


}


