/*
 * Decompiled with CFR 0.152.
 */
package thredds.server.radarServer2;

import java.io.File;
import java.io.IOException;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.EnumMap;
import java.util.Formatter;
import java.util.List;
import java.util.Set;
import java.util.Stack;
import java.util.TimeZone;
import java.util.TreeSet;
import thredds.server.radarServer2.StationList;
import ucar.nc2.constants.FeatureType;
import ucar.nc2.dt.RadialDatasetSweep;
import ucar.nc2.ft.FeatureDatasetFactoryManager;
import ucar.nc2.time.CalendarDate;
import ucar.nc2.time.CalendarDateRange;
import ucar.nc2.time.CalendarPeriod;
import ucar.nc2.units.DateFromString;
import ucar.unidata.geoloc.EarthLocation;

public class RadarDataInventory {
    private EnumMap<DirType, Set<String>> items = new EnumMap(DirType.class);
    private Path root;
    private Path collectionDir;
    private DirectoryStructure structure;
    private String fileTimeFmt;
    private boolean dirty;
    private int maxCrawlItems;
    private int maxCrawlDepth;
    private StationList stations;
    private CalendarPeriod nearestWindow;
    private CalendarPeriod rangeAdjustment;
    private String name;
    private String description;

    public RadarDataInventory(Path datasetRoot, String collection) {
        this.root = datasetRoot;
        this.name = collection;
        this.description = collection;
        this.collectionDir = this.root.resolve(collection);
        this.structure = new DirectoryStructure(this.collectionDir);
        this.dirty = true;
        this.maxCrawlItems = 10;
        this.maxCrawlDepth = 2;
        this.stations = new StationList();
        this.nearestWindow = CalendarPeriod.of((int)1, (CalendarPeriod.Field)CalendarPeriod.Field.Hour);
    }

    public Path getRoot() {
        return this.root;
    }

    public Path getCollectionDir() {
        return this.collectionDir;
    }

    public void setName(String name) {
        this.name = name;
    }

    public String getName() {
        return this.name;
    }

    public String getDescription() {
        return this.description;
    }

    public void setDescription(String description) {
        this.description = description;
    }

    public boolean needsVar() {
        return this.items.containsKey((Object)DirType.Variable);
    }

    public StationList getStationList() {
        return this.stations;
    }

    public void setNearestWindow(CalendarPeriod pd) {
        this.nearestWindow = pd;
    }

    public void addStationDir() {
        this.structure.addSubDir(DirType.Station, null);
        this.dirty = true;
    }

    public void addVariableDir() {
        this.structure.addSubDir(DirType.Variable, null);
        this.dirty = true;
    }

    public void addDateDir(String fmt) {
        this.structure.addSubDir(DirType.Date, fmt);
        CalendarPeriod adjust = this.findRangeAdjustment(fmt);
        if (this.rangeAdjustment == null) {
            this.rangeAdjustment = adjust;
        } else if (adjust.getConvertFactor(this.rangeAdjustment) > 1.0) {
            this.rangeAdjustment = adjust;
        }
        this.dirty = true;
    }

    private CalendarPeriod findRangeAdjustment(String fmtString) {
        if (fmtString.contains("H") || fmtString.contains("k")) {
            return CalendarPeriod.of((int)1, (CalendarPeriod.Field)CalendarPeriod.Field.Hour);
        }
        if (fmtString.contains("d")) {
            return CalendarPeriod.of((int)1, (CalendarPeriod.Field)CalendarPeriod.Field.Day);
        }
        if (fmtString.contains("M")) {
            return CalendarPeriod.of((int)31, (CalendarPeriod.Field)CalendarPeriod.Field.Day);
        }
        return CalendarPeriod.of((int)366, (CalendarPeriod.Field)CalendarPeriod.Field.Day);
    }

    public void addFileTime(String fmt) {
        this.fileTimeFmt = fmt;
    }

    private void findItems(Path start, int level) throws IOException {
        Set<Object> values;
        if (level >= this.structure.order.size() || level >= this.maxCrawlDepth) {
            return;
        }
        DirectoryStructure.DirEntry entry = (DirectoryStructure.DirEntry)this.structure.order.get(level);
        if (!this.items.containsKey((Object)entry.type)) {
            values = new TreeSet();
            this.items.put(entry.type, values);
        } else {
            values = this.items.get((Object)entry.type);
        }
        int crawled = 0;
        for (Path p : Files.newDirectoryStream(start)) {
            if (!p.toFile().isDirectory()) continue;
            String item = p.getFileName().toString();
            values.add(item);
            if (entry.type == DirType.Station) {
                this.updateStations(item, p);
            }
            if (crawled >= this.maxCrawlItems) continue;
            this.findItems(p, level + 1);
            ++crawled;
        }
    }

    private void updateStations(String station, Path path) {
        StationList.Station match = this.stations.get(station);
        if (match == null) {
            try {
                Files.walkFileTree(path, new StationVisitor());
            }
            catch (IOException e) {
                System.out.println("Error walking to find station info: " + station);
            }
        }
    }

    private void update() {
        try {
            if (this.dirty) {
                this.findItems(this.structure.base, 0);
            }
        }
        catch (IOException e) {
            System.out.println("Error updating data inventory.");
        }
    }

    public List<String> listItems(DirType type) {
        this.update();
        Set<String> vals = this.items.get((Object)type);
        if (vals == null) {
            return new ArrayList<String>();
        }
        return new ArrayList<String>((Collection)this.items.get((Object)type));
    }

    public Query newQuery() {
        this.update();
        return new Query();
    }

    public static void main(String[] args) throws IOException {
        for (String name : args) {
            File baseDir = new File(name);
            if (!baseDir.isDirectory()) continue;
            RadarDataInventory dw = new RadarDataInventory(baseDir.toPath(), "");
            dw.addVariableDir();
            dw.addStationDir();
            dw.addDateDir("yyyyMMdd");
            System.out.println("Stations:");
            for (String res : dw.listItems(DirType.Station)) {
                System.out.println("\t" + res);
            }
            System.out.println("Variables:");
            for (String res : dw.listItems(DirType.Variable)) {
                System.out.println("\t" + res);
            }
            System.out.println("Dates:");
            for (String res : dw.listItems(DirType.Date)) {
                System.out.println("\t" + res);
            }
            Query q = dw.newQuery();
            q.addVariable("N0Q");
            q.addStation("TLX");
            q.addDateRange(CalendarDateRange.of((CalendarDate)CalendarDate.of(null, (int)2014, (int)6, (int)24, (int)0, (int)0, (int)0), (CalendarDate)CalendarDate.of(null, (int)2014, (int)6, (int)25, (int)0, (int)0, (int)0)));
            System.out.println("Results of query:");
            for (Query.QueryResultItem i : q.results()) {
                System.out.println("File: " + i.file.toString());
            }
        }
    }

    public class Query {
        private EnumMap<DirType, List<Object>> q = new EnumMap(DirType.class);

        public void addCriteria(DirType type, Object val) {
            List<Object> curVals = this.q.get((Object)type);
            if (curVals == null) {
                curVals = new ArrayList<Object>();
                this.q.put(type, curVals);
            }
            curVals.add(val);
        }

        public void addStation(String stID) {
            this.addCriteria(DirType.Station, stID);
        }

        public void addVariable(String varName) {
            this.addCriteria(DirType.Variable, varName);
        }

        public void addDateRange(CalendarDateRange range) {
            this.addCriteria(DirType.Date, range);
        }

        private CalendarDateRange rangeFromFormat(SimpleDateFormat fmt, CalendarDateRange range) {
            CalendarDate newStart = this.reparseDate(fmt, range.getStart());
            CalendarDate newEnd = this.reparseDate(fmt, range.getEnd());
            return CalendarDateRange.of((CalendarDate)newStart, (CalendarDate)newEnd.add(RadarDataInventory.this.rangeAdjustment));
        }

        private CalendarDate reparseDate(SimpleDateFormat fmt, CalendarDate d) {
            try {
                Date newDate = fmt.parse(fmt.format(d.toDate()));
                return CalendarDate.of((Date)newDate);
            }
            catch (ParseException e) {
                return null;
            }
        }

        private boolean checkDate(CalendarDateRange range, CalendarDate d) {
            return range == null || range.includes(d);
        }

        public List<QueryResultItem> results() throws IOException {
            ArrayList<Path> results = new ArrayList<Path>();
            DirectoryStructure.DirectoryMatcher matcher = RadarDataInventory.this.structure.matcher();
            results.add(RadarDataInventory.this.structure.base);
            for (int i = 0; i < RadarDataInventory.this.structure.order.size(); ++i) {
                DirectoryStructure.DirEntry entry = (DirectoryStructure.DirEntry)RadarDataInventory.this.structure.order.get(i);
                ArrayList<Path> newResults = new ArrayList<Path>();
                List<Object> queryItem = this.q.get((Object)entry.type);
                switch (entry.type) {
                    case Date: {
                        for (Path p : results) {
                            for (Path sub : Files.newDirectoryStream(p)) {
                                newResults.add(sub);
                            }
                        }
                        matcher.push(DirType.Date, entry.fmt);
                        break;
                    }
                    default: {
                        String next = queryItem.get(0).toString();
                        for (Path p : results) {
                            Path nextPath = p.resolve(next);
                            if (!nextPath.toFile().exists()) continue;
                            newResults.add(nextPath);
                        }
                        matcher.push(entry.type, next);
                    }
                }
                results = newResults;
            }
            SimpleDateFormat fmt = new SimpleDateFormat(matcher.matchString());
            fmt.setTimeZone(TimeZone.getTimeZone("UTC"));
            ArrayList<Path> filteredResults = new ArrayList<Path>();
            List<Object> dates = this.q.get((Object)DirType.Date);
            CalendarDateRange range = (CalendarDateRange)dates.get(0);
            if (range.isPoint()) {
                range = CalendarDateRange.of((CalendarDate)range.getStart().subtract(RadarDataInventory.this.nearestWindow), (CalendarDate)range.getEnd().add(RadarDataInventory.this.nearestWindow));
            }
            CalendarDateRange dirRange = this.rangeFromFormat(fmt, range);
            for (Path p : results) {
                try {
                    Date d = fmt.parse(p.toString());
                    if (!this.checkDate(dirRange, CalendarDate.of((Date)d))) continue;
                    filteredResults.add(p);
                }
                catch (ParseException e) {}
            }
            ArrayList<QueryResultItem> filteredFiles = new ArrayList<QueryResultItem>();
            for (Path p : filteredResults) {
                for (Path f : Files.newDirectoryStream(p)) {
                    CalendarDate cd;
                    Date d = DateFromString.getDateUsingDemarkatedMatch((String)f.toString(), (String)RadarDataInventory.this.fileTimeFmt, (char)'#');
                    if (d == null || !this.checkDate(range, cd = CalendarDate.of((Date)d))) continue;
                    filteredFiles.add(new QueryResultItem(f, cd));
                }
            }
            CalendarDateRange originalRange = (CalendarDateRange)dates.get(0);
            if (originalRange.isPoint()) {
                long offset = Long.MAX_VALUE;
                QueryResultItem bestMatch = null;
                for (QueryResultItem it : filteredFiles) {
                    long check = Math.abs(it.time.getDifferenceInMsecs(originalRange.getStart()));
                    if (check >= offset) continue;
                    offset = check;
                    bestMatch = it;
                }
                filteredFiles.clear();
                filteredFiles.add(bestMatch);
            }
            return filteredFiles;
        }

        public class QueryResultItem {
            public Path file;
            public CalendarDate time;

            private QueryResultItem(Path f, CalendarDate cd) {
                this.file = f;
                this.time = cd;
            }
        }
    }

    private class StationVisitor
    extends SimpleFileVisitor<Path> {
        private StationVisitor() {
        }

        @Override
        public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) {
            try {
                RadialDatasetSweep rds = (RadialDatasetSweep)FeatureDatasetFactoryManager.open((FeatureType)FeatureType.RADIAL, (String)file.toString(), null, (Formatter)new Formatter());
                EarthLocation loc = rds.getCommonOrigin();
                StationList.Station added = RadarDataInventory.this.stations.addStation(rds.getRadarID(), loc.getLatLon());
                added.setElevation(loc.getAltitude());
                added.setName(rds.getRadarName());
                return FileVisitResult.TERMINATE;
            }
            catch (IOException e) {
                return FileVisitResult.CONTINUE;
            }
        }
    }

    public class DirectoryStructure {
        private Path base;
        private String sep;
        private List<DirEntry> order;

        public DirectoryStructure(Path dir) {
            this.base = dir;
            this.sep = System.getProperty("file.separator");
            this.order = new ArrayList<DirEntry>();
        }

        public void addSubDir(DirType type, String fmt) {
            this.order.add(new DirEntry(type, fmt));
        }

        public DirectoryMatcher matcher() {
            return new DirectoryMatcher();
        }

        private class DirectoryMatcher {
            Stack<Item> stack = new Stack();
            private boolean inString = true;
            String matcher;

            public DirectoryMatcher() {
                this.matcher = "'" + DirectoryStructure.this.base.toString();
            }

            public void push(DirType type, String value) {
                this.stack.push(new Item(this.matcher, this.inString));
                String newSegment = this.inString ? RadarDataInventory.this.structure.sep : "'" + RadarDataInventory.this.structure.sep;
                if (type == DirType.Date) {
                    newSegment = newSegment + "'";
                    this.inString = false;
                } else {
                    this.inString = true;
                }
                this.matcher = this.matcher + newSegment + value;
            }

            public String matchString() {
                return this.matcher;
            }

            public void pop() {
                Item it = this.stack.pop();
                this.inString = it.inString;
                this.matcher = it.path;
            }

            private class Item {
                String path;
                boolean inString;

                Item(String path, boolean inString) {
                    this.path = path;
                    this.inString = inString;
                }
            }
        }

        private class DirEntry {
            public DirType type;
            public String fmt;

            public DirEntry(DirType type, String fmt) {
                this.type = type;
                this.fmt = fmt;
            }
        }
    }

    public static enum DirType {
        Station,
        Variable,
        Date;

    }
}

