/*
 * Copyright 1998-2014 University Corporation for Atmospheric Research/Unidata
 *
 *   Portions of this software were developed by the Unidata Program at the
 *   University Corporation for Atmospheric Research.
 *
 *   Access and use of this software shall impose the following obligations
 *   and understandings on the user. The user is granted the right, without
 *   any fee or cost, to use, copy, modify, alter, enhance and distribute
 *   this software, and any derivative works thereof, and its supporting
 *   documentation for any purpose whatsoever, provided that this entire
 *   notice appears in all copies of the software, derivative works and
 *   supporting documentation.  Further, UCAR requests that the user credit
 *   UCAR/Unidata in any publications that result from the use of this
 *   software or in any product that includes this software. The names UCAR
 *   and/or Unidata, however, may not be used in any advertising or publicity
 *   to endorse or promote any products or commercial entity unless specific
 *   written permission is obtained from UCAR/Unidata. The user also
 *   understands that UCAR/Unidata is not obligated to provide the user with
 *   any support, consulting, training or assistance of any kind with regard
 *   to the use, operation and performance of this software nor to provide
 *   the user with any updates, revisions, new versions or "bug fixes."
 *
 *   THIS SOFTWARE IS PROVIDED BY UCAR/UNIDATA "AS IS" AND ANY EXPRESS OR
 *   IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 *   WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 *   DISCLAIMED. IN NO EVENT SHALL UCAR/UNIDATA BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE ACCESS, USE OR PERFORMANCE OF THIS SOFTWARE.
 */
package thredds.server.opendap;

import net.jcip.annotations.Immutable;
import opendap.servlet.GuardedDataset;
import ucar.nc2.NetcdfFile;

import java.io.IOException;

/**
 * Regenerate DDS, DAS each time
 */
@Immutable
public class GuardedDatasetImpl implements GuardedDataset {
  static protected org.slf4j.Logger log = org.slf4j.LoggerFactory.getLogger(GuardedDataset.class);

  private final boolean hasSession;
  private final NetcdfFile org_file;
  private final String reqPath;
  private boolean closed = false;

  public void release() {
    if (!hasSession)
      close();
    closed = true;
  }

  public void close() {
    try {
      org_file.close();
    } catch (IOException e) {
      log.error("GuardedDatasetImpl close", e);
    }
  }

  public GuardedDatasetImpl(String reqPath, NetcdfFile ncfile, boolean hasSession) {
    this.org_file = ncfile;
    this.reqPath = reqPath;
    this.hasSession = hasSession;
  }

  public opendap.servers.ServerDDS getDDS() {
    if (closed) throw new IllegalStateException("getDDS(): "+this + " closed");
    return new NcDDS(reqPath, org_file);
  }

  public opendap.dap.DAS getDAS() {
    if (closed) throw new IllegalStateException("getDAS(): "+this + " closed");
    return new NcDAS(org_file);
  }

  public String toString() {
    String name = org_file.getLocation();
    return name == null ? org_file.getCacheName() : name;
  }
}