/*
 * Copyright (c) 1998-2018 John Caron and University Corporation for Atmospheric Research/Unidata
 * See LICENSE for license information.
 */

package thredds.server.admin;

import java.io.*;
import java.util.*;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import com.coverity.security.Escape;
import com.google.common.escape.Escaper;
import com.google.common.eventbus.EventBus;
import com.google.common.net.UrlEscapers;
import org.quartz.JobKey;
import org.quartz.TriggerKey;
import org.quartz.impl.matchers.GroupMatcher;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import thredds.core.DataRootManager;
import thredds.core.DatasetManager;
import thredds.featurecollection.CollectionUpdater;
import thredds.featurecollection.InvDatasetFeatureCollection;
import thredds.featurecollection.FeatureCollectionConfig;
import thredds.featurecollection.FeatureCollectionType;
import thredds.inventory.*;
import thredds.server.catalog.FeatureCollectionRef;
import thredds.server.config.TdsContext;
import thredds.servlet.ServletUtil;
import thredds.util.ContentType;
import ucar.nc2.grib.collection.GribCdmIndex;
import ucar.nc2.util.IO;
import ucar.unidata.util.StringUtil2;

/**
 * Allow external triggers for rereading Feature collections
 *
 * @author caron
 * @since May 4, 2010
 */
@Controller
@RequestMapping(value = {"/admin/collection"})
public class AdminCollectionController implements InitializingBean {
  private static final org.slf4j.Logger log = org.slf4j.LoggerFactory.getLogger(AdminCollectionController.class);

  private static final String PATH = "/admin/collection";
  private static final String COLLECTION = "collection";
  private static final String SHOW_COLLECTION = "showCollection";
  private static final String SHOW = "showStatus";
  private static final String SHOW_CSV = "showStatus.csv";
  private static final String DOWNLOAD = "download";
  private static final String DOWNLOAD_ALL = "downloadAll";
  private static final String TRIGGER = "trigger";

  @Autowired
  DebugCommands debugCommands;

  @Autowired
  DataRootManager dataRootManager;

  @Autowired
  private TdsContext tdsContext;

  @Autowired
  private DatasetManager datasetManager;

  @Autowired
  @Qualifier("fcTriggerEventBus")
  private EventBus eventBus;

  @Autowired
  CollectionUpdater collectionUpdater;

  public void afterPropertiesSet() {
    Escaper urlParamEscaper = UrlEscapers.urlFormParameterEscaper();

    DebugCommands.Category debugHandler = debugCommands.findCategory("Collections");
    DebugCommands.Action act;

    act = new DebugCommands.Action("showCollection", "Show Collections") {
      public void doAction(DebugCommands.Event e) {
        // get sorted list of collections
        List<FeatureCollectionRef> fcList = dataRootManager.getFeatureCollections();
        Collections.sort(fcList, (o1, o2) -> o1.getCollectionName().compareTo(o2.getCollectionName()));

        for (FeatureCollectionRef fc : fcList) {
          String uriParam = Escape.uriParam(fc.getCollectionName());
          String url = tdsContext.getContextPath() + PATH + "/" + SHOW_COLLECTION + "?" + COLLECTION + "=" + uriParam;
          e.pw.printf("<p/><a href='%s'>%s</a> (%s)%n", url, fc.getCollectionName(), fc.getName());
          FeatureCollectionConfig config = fc.getConfig();
          if (config != null)
            e.pw.printf("%s%n", config.spec);
        }

        String url = tdsContext.getContextPath() + PATH + "/" + SHOW;
        e.pw.printf("<p/><a href='%s'>Show All Collection Status</a>%n", url);

        url = tdsContext.getContextPath() + PATH + "/" + SHOW_CSV;
        e.pw.printf("<p/><a href='%s'>Collection Status CSV</a>%n", url);

        url = tdsContext.getContextPath() + PATH + "/" + DOWNLOAD_ALL;
        e.pw.printf("<p/><a href='%s'>Download All top-level collection indices</a>%n", url);
      }
    };
    debugHandler.addAction(act);

    act = new DebugCommands.Action("sched", "Show FeatureCollection update scheduler") {
      public void doAction(DebugCommands.Event e) {
        org.quartz.Scheduler scheduler = collectionUpdater.getScheduler();
        if (scheduler == null)
          return;

        try {
          e.pw.println(scheduler.getMetaData());

          List<String> groups = scheduler.getJobGroupNames();
          List<String> triggers = scheduler.getTriggerGroupNames();

          // enumerate each job group
          for (String group : scheduler.getJobGroupNames()) {
            e.pw.println("Group " + group);

            // enumerate each job in group
            for (JobKey jobKey : scheduler.getJobKeys(GroupMatcher.<JobKey>groupEquals(group))) {
              e.pw.println("  Job " + jobKey.getName());
              e.pw.println("    " + scheduler.getJobDetail(jobKey));
            }

            // enumerate each trigger in group
            for (TriggerKey triggerKey : scheduler.getTriggerKeys(GroupMatcher.<TriggerKey>groupEquals(group))) {
              e.pw.println("  Trigger " + triggerKey.getName());
              e.pw.println("    " + scheduler.getTrigger(triggerKey));
            }
          }
        } catch (Exception e1) {
          e.pw.println("Error on scheduler " + e1.getMessage());
          log.error("Error on scheduler " + e1.getMessage());
        }
      }
    };
    debugHandler.addAction(act);

  }

  @RequestMapping(value = "/" + SHOW_COLLECTION, method = RequestMethod.GET)
  protected ResponseEntity<String> showCollection(@RequestParam String collection) throws Exception {
    Formatter out = new Formatter();

    FeatureCollectionRef want = dataRootManager.findFeatureCollection(collection);

    HttpStatus status = HttpStatus.OK;
    if (want == null) {
      status = HttpStatus.NOT_FOUND;
      out.format("NOT FOUND");

    } else {
      out.format("<h3>Collection %s</h3>%n%n", Escape.html(collection));
      showFeatureCollection(out, want);

      String uriParam = Escape.uriParam(want.getCollectionName());
      String url = tdsContext.getContextPath() + PATH + "/" + TRIGGER + "?" + COLLECTION + "=" + uriParam + "&"
          + TRIGGER + "=" + CollectionUpdateType.nocheck;
      out.format("<p/><a href='%s'>Send trigger to %s</a>%n", url, Escape.html(want.getCollectionName()));

      String url2 = tdsContext.getContextPath() + PATH + "/" + DOWNLOAD + "?" + COLLECTION + "=" + uriParam;
      out.format("<p/><a href='%s'>Download index file for %s</a>%n", url2, Escape.html(want.getCollectionName()));
    }

    HttpHeaders responseHeaders = new HttpHeaders();
    responseHeaders.setContentType(MediaType.TEXT_HTML);
    return new ResponseEntity<>(out.toString(), responseHeaders, status);
  }

  @RequestMapping(value = {"/" + SHOW})
  protected ResponseEntity<String> showCollectionStatus() throws Exception {
    Formatter out = new Formatter();

    // get sorted list of collections
    List<FeatureCollectionRef> fcList = dataRootManager.getFeatureCollections();
    Collections.sort(fcList, (o1, o2) -> o1.getCollectionName().compareTo(o2.getCollectionName()));

    for (FeatureCollectionRef fc : fcList) {
      String uriParam = Escape.uriParam(fc.getCollectionName());
      String url = tdsContext.getContextPath() + PATH + "?" + COLLECTION + "=" + uriParam;
      out.format("<p/><a href='%s'>%s</a> (%s)%n", url, fc.getCollectionName(), fc.getName());
      InvDatasetFeatureCollection fcd = datasetManager.openFeatureCollection(fc);

      out.format("<pre>%s</pre>%n", fcd.showStatusShort("txt"));
    }

    HttpHeaders responseHeaders = new HttpHeaders();
    responseHeaders.setContentType(MediaType.TEXT_HTML);
    return new ResponseEntity<>(out.toString(), responseHeaders, HttpStatus.OK);
  }

  @RequestMapping(value = {"/" + SHOW_CSV})
  protected ResponseEntity<String> showCollectionStatusCsv() throws Exception {
    Formatter out = new Formatter();

    // get sorted list of collections
    List<FeatureCollectionRef> fcList = dataRootManager.getFeatureCollections();
    Collections.sort(fcList, (o1, o2) -> {
      int compareType = o1.getConfig().type.toString().compareTo(o1.getConfig().type.toString());
      if (compareType != 0)
        return compareType;
      return o1.getCollectionName().compareTo(o2.getCollectionName());
    });

    out.format("%s, %s, %s, %s, %s, %s, %s, %s, %s%n", "collection", "ed", "type", "group", "nrecords", "ndups", "%",
        "nmiss", "%");
    for (FeatureCollectionRef fc : fcList) {
      if (fc.getConfig().type != FeatureCollectionType.GRIB1 && fc.getConfig().type != FeatureCollectionType.GRIB2)
        continue;
      InvDatasetFeatureCollection fcd = datasetManager.openFeatureCollection(fc);
      out.format("%s", fcd.showStatusShort("csv"));
    }

    HttpHeaders responseHeaders = new HttpHeaders();
    responseHeaders.setContentType(MediaType.TEXT_PLAIN);
    return new ResponseEntity<>(out.toString(), responseHeaders, HttpStatus.OK);
  }

  @RequestMapping(value = {"/trigger"}) // LOOK should require collection and trigger type params
  protected ResponseEntity<String> triggerFeatureCollection(HttpServletRequest req, HttpServletResponse res)
      throws Exception {
    Formatter out = new Formatter();

    CollectionUpdateType triggerType = null;
    String triggerTypeS = req.getParameter(TRIGGER);
    try {
      triggerType = CollectionUpdateType.valueOf(triggerTypeS);
    } catch (Throwable t) {
      // noop
    }

    if (triggerType == null) {
      res.setStatus(HttpServletResponse.SC_BAD_REQUEST);
      out.format(" TRIGGER Type '%s' not legal%n", Escape.html(triggerTypeS));
      return null;
    }

    String collectName = StringUtil2.unescape(req.getParameter(COLLECTION)); // this is the collection name
    FeatureCollectionRef want = dataRootManager.findFeatureCollection(collectName);

    if (want == null) {
      res.setStatus(HttpServletResponse.SC_NOT_FOUND);
      out.format("NOT FOUND");

    } else {
      out.format("<h3>Collection %s</h3>%n", Escape.html(collectName));

      if (!want.getConfig().isTrigggerOk()) {
        res.setStatus(HttpServletResponse.SC_FORBIDDEN);
        out.format(" TRIGGER NOT ENABLED%n");

      } else {
        eventBus.post(new CollectionUpdateEvent(triggerType, collectName, "trigger"));
        // CollectionUpdater.INSTANCE.triggerUpdate(collectName, triggerType);
        out.format(" TRIGGER SENT%n");
      }
    }

    HttpHeaders responseHeaders = new HttpHeaders();
    responseHeaders.setContentType(MediaType.TEXT_HTML);
    return new ResponseEntity<>(out.toString(), responseHeaders, HttpStatus.OK);
  }

  @RequestMapping(value = {"/" + DOWNLOAD_ALL})
  protected void downloadAll(HttpServletRequest req, HttpServletResponse res) throws IOException {
    File tempFile = File.createTempFile("CollectionIndex", ".zip");
    try (FileOutputStream fos = new FileOutputStream(tempFile.getPath())) {
      ZipOutputStream zout = new ZipOutputStream(fos);
      for (FeatureCollectionRef fc : dataRootManager.getFeatureCollections()) {
        File idxFile = GribCdmIndex.getTopIndexFileFromConfig(fc.getConfig());
        if (idxFile == null)
          continue;
        ZipEntry entry = new ZipEntry(idxFile.getName());
        zout.putNextEntry(entry);
        IO.copyFile(idxFile.getPath(), zout);
        zout.closeEntry();
      }
      zout.close();
      fos.close();
    }

    ServletUtil.returnFile(req, res, tempFile, ContentType.binary.toString());
    // tempFile.delete();
  }

  @RequestMapping(value = {"/" + DOWNLOAD})
  protected ResponseEntity<String> downloadIndex(HttpServletRequest req, HttpServletResponse res) throws Exception {
    String collectName = StringUtil2.unescape(req.getParameter(COLLECTION)); // this is the collection name
    FeatureCollectionRef want = dataRootManager.findFeatureCollection(collectName);

    if (want == null) {
      HttpHeaders responseHeaders = new HttpHeaders();
      responseHeaders.setContentType(MediaType.TEXT_PLAIN);
      return new ResponseEntity<>(Escape.html(collectName) + " NOT FOUND", responseHeaders, HttpStatus.NOT_FOUND);
    }

    File idxFile = GribCdmIndex.getTopIndexFileFromConfig(want.getConfig());
    if (idxFile == null) {
      HttpHeaders responseHeaders = new HttpHeaders();
      responseHeaders.setContentType(MediaType.TEXT_PLAIN);
      return new ResponseEntity<>(Escape.html(collectName) + " NOT FOUND", responseHeaders, HttpStatus.NOT_FOUND);
    }

    ServletUtil.returnFile(req, res, idxFile, ContentType.binary.toString());
    return null;
  }

  private void showFeatureCollection(Formatter out, FeatureCollectionRef fc) throws IOException {
    FeatureCollectionConfig config = fc.getConfig(); // LOOK
    if (config != null) {
      Formatter f = new Formatter();
      config.show(f);
      out.format("%n<pre>%s%n</pre>", f.toString());
    }

    InvDatasetFeatureCollection fcd = datasetManager.openFeatureCollection(fc);
    out.format("%n<pre>%n");
    fcd.showStatus(out);
    out.format("%n</pre>%n");
  }
}
