/*
 * Copyright 1998-2014 University Corporation for Atmospheric Research/Unidata
 *
 *   Portions of this software were developed by the Unidata Program at the
 *   University Corporation for Atmospheric Research.
 *
 *   Access and use of this software shall impose the following obligations
 *   and understandings on the user. The user is granted the right, without
 *   any fee or cost, to use, copy, modify, alter, enhance and distribute
 *   this software, and any derivative works thereof, and its supporting
 *   documentation for any purpose whatsoever, provided that this entire
 *   notice appears in all copies of the software, derivative works and
 *   supporting documentation.  Further, UCAR requests that the user credit
 *   UCAR/Unidata in any publications that result from the use of this
 *   software or in any product that includes this software. The names UCAR
 *   and/or Unidata, however, may not be used in any advertising or publicity
 *   to endorse or promote any products or commercial entity unless specific
 *   written permission is obtained from UCAR/Unidata. The user also
 *   understands that UCAR/Unidata is not obligated to provide the user with                                   Time
 *   any support, consulting, training or assistance of any kind with regard
 *   to the use, operation and performance of this software nor to provide
 *   the user with any updates, revisions, new versions or "bug fixes."
 *
 *   THIS SOFTWARE IS PROVIDED BY UCAR/UNIDATA "AS IS" AND ANY EXPRESS OR
 *   IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 *   WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 *   DISCLAIMED. IN NO EVENT SHALL UCAR/UNIDATA BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE ACCESS, USE OR PERFORMANCE OF THIS SOFTWARE.
 */

/* Generated By:JavaCC: Do not edit this line. StandardUnitFormat.java */
package ucar.units;

import java.io.InputStreamReader;
import java.io.LineNumberReader;
import java.io.StringReader;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Comparator;
import java.util.Date;
import java.util.Locale;
import java.util.TimeZone;

/**
 * Standard formatter/parser for unit specifications.
 * <p/>
 * Instances of this class are thread-compatible but not thread-safe.
 *
 * @author Steven R. Emmerson
 */
public final class
        StandardUnitFormat
        extends UnitFormatImpl implements StandardUnitFormatConstants {
  private static final long serialVersionUID = 2L;

  /**
   * The singleton instance of this class.
   *
   * @serial
   */
  private static StandardUnitFormat _instance;

  /**
   * The date formatter.
   *
   * @serial
   */
  private static final SimpleDateFormat dateFormat;

  /**
   * The Comparator for ordering base units for printing.  Orders
   * Factor-s by decreasing exponent (major) and lexically (minor).
   *
   * @serial
   */
  private static final Comparator<Factor> factorComparator =
          new Comparator<Factor>() {
            public int compare(Factor f1, Factor f2) {
              int comp = f2.getExponent() - f1.getExponent();
              if (comp == 0)
                comp = f1.getID().compareTo(f2.getID());
              return comp;
            }
          };

  static {
    dateFormat =
            (SimpleDateFormat) DateFormat.getDateInstance(
                    DateFormat.SHORT, Locale.US);
    dateFormat.setTimeZone(TimeZone.getTimeZone("UTC"));
    dateFormat.applyPattern(" '@' yyyy-MM-dd HH:mm:ss.SSS 'UTC'");
  }

  /**
   * Constructs from nothing.
   */
  private StandardUnitFormat() {
    this(new StringReader(""));
  }


  /**
   * Returns an instance of this class.
   *
   * @return An instance of this class.
   */
  public static synchronized StandardUnitFormat instance() {
    if (_instance == null)
      _instance = new StandardUnitFormat();
    return _instance;
  }


  /**
   * Indicates if a unit is a time unit.
   *
   * @param unit The unit in question.
   * @return {@code true} if and only if {@code unit} is a time unit.
   * @Throws UnitSystemException if the unit system can't be initialized.
   */
  private static boolean isTimeUnit(final Unit unit) throws UnitSystemException {
    return unit.isCompatible(UnitSystemManager.instance().getBaseUnit(
            BaseQuantity.TIME));
  }


  /**
   * Decodes a unit specification.  An unrecognized unit is made into
   * an UnknownUnit.
   *
   * @param spec   The unit specification to be decoded.
   * @param unitDB The unit database to use.
   * @return The unit corresponding to the specification.
   * @throws UnitParseException     The unit specification syntax is
   *                                invalid.
   * @throws SpecificationException Something's wrong with the
   *                                specification.
   * @throws UnitDBException        Something's wrong with the unit
   *                                database.
   * @throws PrefixDBException      Something's wrong with the unit prefix
   *                                database.
   * @throws UnitSystemException    Something's wrong with the underlying
   *                                system of units.
   */
  public Unit
  parse(String spec, UnitDB unitDB)
          throws UnitParseException,
          SpecificationException,
          UnitDBException,
          PrefixDBException,
          UnitSystemException {
    if (spec == null)
      throw new UnitParseException(spec);

    ReInit(new StringReader(spec.trim()));

    try {
      Unit unit = unitSpec(unitDB);
      return unit;
    } catch (TokenMgrError e) {
      throw new UnitParseException(spec, e);
    } catch (ParseException e) {
      throw new UnitParseException(spec, e);
    } catch (OperationException e) {
      throw new SpecificationException(spec, e);
    }
  }


  /**
   * Formats a Factor.
   *
   * @param factor The factor to be formatted.
   * @param buf    The buffer to append to.
   * @return The appended-to buffer.
   */
  public StringBuffer
  format(Factor factor, StringBuffer buf) {
    return buf.append(factor.toString());
  }


  /**
   * Formats a unit.  The symbol or name will be used if available;
   * otherwise, a specification in terms of underlying units will be
   * returned.
   *
   * @param unit The unit to be formatted.
   * @param buf  The buffer to append to.
   * @return The appended-to buffer.
   * @throws UnitClassException The class of the unit is unknown.
   */
  public StringBuffer
  format(Unit unit, StringBuffer buf)
          throws UnitClassException {
    return format(unit, buf, true);
  }


  /**
   * Formats a unit in the underlying system of units.
   *
   * @param unit The unit to be formatted.
   * @param buf  The buffer to append to.
   * @return The appended-to buffer.
   * @throws UnitClassException The class of the unit is unknown.
   */
  public StringBuffer
  longFormat(Unit unit, StringBuffer buf)
          throws UnitClassException {
    return format(unit, buf, false);
  }


  /**
   * Formats a unit.
   *
   * @param unit      The unit to be formatted.
   * @param buf       The buffer to append to.
   * @param normalize Whether or not to reduce the unit.
   * @return The appended-to buffer.
   * @throws UnitClassException The class of the unit is unknown.
   */
  private StringBuffer
  format(Unit unit, StringBuffer buf, boolean normalize)
          throws UnitClassException {
    boolean done = false;
    if (!normalize) {
      String id = unit.getSymbol();
      if (id == null)
        id = unit.getName();
      if (id != null) {
        buf.append(id.replace(' ', '_'));
        done = true;
      }
    }
    if (!done) {
      if (unit instanceof BaseUnit)
        format((BaseUnit) unit, buf);
      else if (unit instanceof DerivedUnit)
        format((DerivedUnit) unit, buf);
      else if (unit instanceof ScaledUnit)
        format((ScaledUnit) unit, buf, normalize);
      else if (unit instanceof OffsetUnit)
        format((OffsetUnit) unit, buf, normalize);
      else if (unit instanceof TimeScaleUnit)
        format((TimeScaleUnit) unit, buf, normalize);
      else
        throw new UnitClassException(unit);
    }
    return buf;
  }


  private StringBuffer
  format(BaseUnit baseUnit, StringBuffer buf) {
    return buf.append(baseUnit.getSymbol());
  }


  private StringBuffer
  format(DerivedUnit unit, StringBuffer buf) {
    Factor[] factors = unit.getDimension().getFactors();
    Arrays.sort(factors, factorComparator);
    for (int i = 0; i < factors.length; i++)
      format(factors[i], buf).append('.');
    if (factors.length != 0)
      buf.setLength(buf.length() - 1);
    return buf;
  }


  private StringBuffer
  format(ScaledUnit unit, StringBuffer buf, boolean normalize)
          throws UnitClassException {
    double scale = unit.getScale();
    if (scale != 0.0) {
      if (scale == 1) {
        format(unit.getUnit(), buf, normalize);
      } else {
        buf.append(scale).append(' ');
        int start = buf.length();
        format(unit.getUnit(), buf, normalize);
        if (start == buf.length())
          buf.setLength(start - 1);
      }
    }
    return buf;
  }


  private StringBuffer
  format(OffsetUnit unit, StringBuffer buf, boolean normalize)
          throws UnitClassException {
    double offset = unit.getOffset();
    if (offset == 0.0) {
      format(unit.getUnit(), buf, normalize);
    } else {
      int start = buf.length();
      format(unit.getUnit(), buf, normalize);
      return (isBlackSpace(buf, start)
              ? buf
              : buf.insert(start, '(').append(')')).
              append(" @ ").append(offset);
    }
    return buf;
  }


  private static boolean
  isBlackSpace(StringBuffer buf, int start) {
    return buf.substring(start).indexOf(' ') == -1;
  }


  private StringBuffer
  format(TimeScaleUnit unit, StringBuffer buf, boolean normalize)
          throws UnitClassException {
    return format(unit.getUnit(), buf, normalize).
            append(dateFormat.format(unit.getOrigin()));
  }


  /**
   * Gets a unit from a unit database.
   */
  private static Unit
  getUnit(UnitDB unitDB, String string)
          throws UnitDBAccessException {
    return unitDB.get(string);
  }


  /**
   * Gets a prefix from the prefix database.
   */
  private static Prefix
  getPrefix(String string)
          throws PrefixDBException {
    PrefixDB prefixDB = PrefixDBManager.instance();
    Prefix prefix = prefixDB.getPrefixByName(string);
    if (prefix == null)
      prefix = prefixDB.getPrefixBySymbol(string);
    return prefix;
  }


  private static void myAssert(StandardUnitFormat parser, final String spec,
                               final Unit unit) throws NoSuchUnitException,
          UnitParseException, SpecificationException, UnitDBException,
          PrefixDBException, UnitSystemException {
    if (!parser.parse(spec).equals(unit)) {
      throw new AssertionError(spec + " != " + unit);
    }
    System.out.println(spec + " -> " + unit);
  }


  /**
   * Test this class.
   */
  public static void main(String[] args)
          throws Exception {
    StandardUnitFormat parser =
            StandardUnitFormat.instance();
    final Unit m = parser.parse("m");
    final Unit s = parser.parse("s");
    final Unit epoch = parser.parse("s @ 1970-01-01 00:00:00 UTC");
    myAssert(parser, "m m", m.multiplyBy(m));
    myAssert(parser, "m.m", m.multiplyBy(m));
    myAssert(parser, "(m)(m)", m.multiplyBy(m));
    myAssert(parser, "m/s/s", m.divideBy(s).divideBy(s));
    myAssert(parser, "m2", m.raiseTo(2));
    myAssert(parser, "m2.s", m.raiseTo(2).multiplyBy(s));
    myAssert(parser, "m2/s", m.raiseTo(2).divideBy(s));
    myAssert(parser, "m^2/s", m.raiseTo(2).divideBy(s));
    myAssert(parser, "m s @ 5", m.multiplyBy(s).shiftTo(5.0));
    myAssert(parser, "m2 s @ 5", m.raiseTo(2).multiplyBy(s).shiftTo(5));
    myAssert(parser, "m2 s-1 @ 5", m.raiseTo(2).multiplyBy(s.raiseTo(-1))
            .shiftTo(5));
    myAssert(parser, "m s from 5", m.multiplyBy(s).shiftTo(5));
    myAssert(parser, "s@19700101", epoch);
    myAssert(parser, "s@19700101T000000", epoch);
    myAssert(parser, "s@19700101T000000.00", epoch);
    myAssert(parser, "s @ 1970-01-01T00:00:00.00", epoch);
    myAssert(parser, "s @ 1970-01-01 00:00:00.00", epoch);
    myAssert(parser, "s @ 1970-01-01 00:00:00.00 +0", epoch);
    myAssert(parser, "s @ 1970-01-01T00:00:00.00 -12", epoch
            .shiftTo(new Date(12 * 60 * 60 * 1000)));
    if (!parser.parse("days since 2009-06-14 04:00:00").equals(
            parser.parse("days since 2009-06-14 04:00:00 +00:00"))) {
      throw new AssertionError();
    }
    myAssert(parser, "lg(re: 1)", DerivedUnitImpl.DIMENSIONLESS.log(10));
    myAssert(parser, "0.1 lg(re 1 mm)", m.multiplyBy(1e-3).log(10)
            .multiplyBy(0.1));
    myAssert(parser, "m", m);
    myAssert(parser, "2 m s", m.multiplyBy(s).multiplyBy(2));
    myAssert(parser, "3.14 m.s", m.multiplyBy(s).multiplyBy(3.14));
    myAssert(parser, "1e9 (m)", m.multiplyBy(1e9));
    myAssert(parser, "(m s)2", m.multiplyBy(s).raiseTo(2));
    myAssert(parser, "m2.s-1", m.raiseTo(2).divideBy(s));
    myAssert(parser, "m2 s^-1", m.raiseTo(2).divideBy(s));
    myAssert(parser, "(m/s)2", m.divideBy(s).raiseTo(2));
    myAssert(parser, "m2/s-1", m.raiseTo(2).divideBy(s.raiseTo(-1)));
    myAssert(parser, "m2/s^-1", m.raiseTo(2).divideBy(s.raiseTo(-1)));
    myAssert(parser, ".5 m/(.25 s)2", m.multiplyBy(.5).divideBy(
            s.multiplyBy(.25).raiseTo(2)));
    myAssert(parser, "m.m-1.m", m.multiplyBy(m.raiseTo(-1)).multiplyBy(m));
    myAssert(parser, "2.0 m 1/2 s-1*(m/s^1)^-1 (1e9 m-1)(1e9 s-1)-1.m/s", m
            .multiplyBy(2).multiplyBy(1. / 2.).multiplyBy(
                    s.raiseTo(-1)).multiplyBy(
                    m.divideBy(s.raiseTo(1)).raiseTo(-1))
            .multiplyBy(m.raiseTo(-1).multiplyBy(1e9)).multiplyBy(
                    s.raiseTo(-1).multiplyBy(1e9).raiseTo(-1))
            .multiplyBy(m).divideBy(s));
    myAssert(parser, "m/km", m.divideBy(m.multiplyBy(1e3)));

    LineNumberReader lineInput = new LineNumberReader(
            new InputStreamReader(System.in));

    for (; ; ) {
      System.out.print("Enter a unit specification or ^D to quit: ");

      String spec = lineInput.readLine();
      if (spec == null)
        break;

      try {
        System.out.println(parser.parse(spec.trim()));
      } catch (Exception e) {
        System.out.println(e.getMessage());
      }
    }
    System.out.println("");
  }

  final public Unit unitSpec(UnitDB unitDB) throws ParseException, OperationException, UnitSystemException, PrefixDBException, UnitDBException {
    Unit unit = DerivedUnitImpl.DIMENSIONLESS;
    switch ((jj_ntk == -1) ? jj_ntk() : jj_ntk) {
      case PLUS:
      case MINUS:
      case UINT:
      case LPAREN:
      case PERIOD:
      case SYMBOL:
      case T:
      case NAME:
      case LB:
      case LN:
      case LG:
        unit = shiftExpr(unitDB);
        break;
      default:
        jj_la1[0] = jj_gen;
        ;
    }
    jj_consume_token(0);
    {
      if (true) return unit;
    }
    throw new Error("Missing return statement in function");
  }

  final public Unit shiftExpr(UnitDB unitDB) throws ParseException, OperationException, UnitSystemException, PrefixDBException, UnitDBException {
    Unit unit;
    Date timestamp;
    double origin;
    unit = productExpr(unitDB);
    switch ((jj_ntk == -1) ? jj_ntk() : jj_ntk) {
      case SHIFT:
        jj_consume_token(SHIFT);
        if (isTimeUnit(unit)) {
          timestamp = timeOriginExpr();
          unit = unit.shiftTo(timestamp);
        } else {
          switch ((jj_ntk == -1) ? jj_ntk() : jj_ntk) {
            case PLUS:
            case MINUS:
            case UINT:
            case PERIOD:
              origin = number();
              unit = unit.shiftTo(origin);
              break;
            default:
              jj_la1[1] = jj_gen;
              jj_consume_token(-1);
              throw new ParseException();
          }
        }
        break;
      default:
        jj_la1[2] = jj_gen;
        ;
    }
    {
      if (true) return unit;
    }
    throw new Error("Missing return statement in function");
  }

  final public Unit productExpr(UnitDB unitDB) throws ParseException, OperationException, UnitSystemException, PrefixDBException, UnitDBException {
    Unit unit;
    Unit unit2;
    unit = powerExpr(unitDB);
    label_1:
    while (true) {
      switch ((jj_ntk == -1) ? jj_ntk() : jj_ntk) {
        case SP:
        case PLUS:
        case MINUS:
        case UINT:
        case LPAREN:
        case PERIOD:
        case STAR:
        case DIVIDE:
        case SYMBOL:
        case T:
        case NAME:
        case LB:
        case LN:
        case LG:
          ;
          break;
        default:
          jj_la1[3] = jj_gen;
          break label_1;
      }
      switch ((jj_ntk == -1) ? jj_ntk() : jj_ntk) {
        case DIVIDE:
          jj_consume_token(DIVIDE);
          unit2 = powerExpr(unitDB);
          unit = unit.divideBy(unit2);
          break;
        default:
          jj_la1[6] = jj_gen;
          if (jj_2_1(2)) {
            switch ((jj_ntk == -1) ? jj_ntk() : jj_ntk) {
              case SP:
              case PERIOD:
              case STAR:
                switch ((jj_ntk == -1) ? jj_ntk() : jj_ntk) {
                  case PERIOD:
                    jj_consume_token(PERIOD);
                    break;
                  case STAR:
                    jj_consume_token(STAR);
                    break;
                  case SP:
                    jj_consume_token(SP);
                    break;
                  default:
                    jj_la1[4] = jj_gen;
                    jj_consume_token(-1);
                    throw new ParseException();
                }
                break;
              default:
                jj_la1[5] = jj_gen;
                ;
            }
            unit2 = powerExpr(unitDB);
            unit = unit.multiplyBy(unit2);
          } else {
            jj_consume_token(-1);
            throw new ParseException();
          }
      }
    }
    {
      if (true) return unit;
    }
    throw new Error("Missing return statement in function");
  }

  final public Unit powerExpr(UnitDB unitDB) throws ParseException, OperationException, UnitSystemException, PrefixDBException, UnitDBException {
    Unit unit;
    int exponent;
    unit = basicExpr(unitDB);
    switch ((jj_ntk == -1) ? jj_ntk() : jj_ntk) {
      case PLUS:
      case MINUS:
      case UINT:
      case RAISE:
        switch ((jj_ntk == -1) ? jj_ntk() : jj_ntk) {
          case RAISE:
            jj_consume_token(RAISE);
            break;
          default:
            jj_la1[7] = jj_gen;
            ;
        }
        exponent = integer();
        unit = unit.raiseTo(exponent);
        break;
      default:
        jj_la1[8] = jj_gen;
        ;
    }
    {
      if (true) return unit;
    }
    throw new Error("Missing return statement in function");
  }

  final public Unit basicExpr(UnitDB unitDB) throws ParseException, OperationException, UnitSystemException, PrefixDBException, UnitDBException {
    Unit unit;
    double number;
    switch ((jj_ntk == -1) ? jj_ntk() : jj_ntk) {
      case PLUS:
      case MINUS:
      case UINT:
      case PERIOD:
        number = number();
        unit = DerivedUnitImpl.DIMENSIONLESS.multiplyBy(number);
        break;
      case SYMBOL:
      case T:
      case NAME:
        unit = unitIdentifier(unitDB);
        break;
      case LB:
      case LN:
      case LG:
        unit = logExpr(unitDB);
        break;
      case LPAREN:
        jj_consume_token(LPAREN);
        switch ((jj_ntk == -1) ? jj_ntk() : jj_ntk) {
          case SP:
            jj_consume_token(SP);
            break;
          default:
            jj_la1[9] = jj_gen;
            ;
        }
        unit = shiftExpr(unitDB);
        switch ((jj_ntk == -1) ? jj_ntk() : jj_ntk) {
          case SP:
            jj_consume_token(SP);
            break;
          default:
            jj_la1[10] = jj_gen;
            ;
        }
        jj_consume_token(RPAREN);
        break;
      default:
        jj_la1[11] = jj_gen;
        jj_consume_token(-1);
        throw new ParseException();
    }
    {
      if (true) return unit;
    }
    throw new Error("Missing return statement in function");
  }

  final public Unit logExpr(UnitDB unitDB) throws ParseException, OperationException, UnitSystemException, PrefixDBException, UnitDBException {
    double base;
    Unit ref;
    switch ((jj_ntk == -1) ? jj_ntk() : jj_ntk) {
      case LB:
        jj_consume_token(LB);
        base = 2;
        break;
      case LN:
        jj_consume_token(LN);
        base = Math.E;
        break;
      case LG:
        jj_consume_token(LG);
        base = 10;
        break;
      default:
        jj_la1[12] = jj_gen;
        jj_consume_token(-1);
        throw new ParseException();
    }
    ref = productExpr(unitDB);
    switch ((jj_ntk == -1) ? jj_ntk() : jj_ntk) {
      case SP:
        jj_consume_token(SP);
        break;
      default:
        jj_la1[13] = jj_gen;
        ;
    }
    jj_consume_token(RPAREN);
    {
      if (true) return ref.log(base);
    }
    throw new Error("Missing return statement in function");
  }

  final public double number() throws ParseException {
    double number;
    if (jj_2_2(2147483647)) {
      number = real();
    } else {
      switch ((jj_ntk == -1) ? jj_ntk() : jj_ntk) {
        case PLUS:
        case MINUS:
        case UINT:
          number = integer();
          break;
        default:
          jj_la1[14] = jj_gen;
          jj_consume_token(-1);
          throw new ParseException();
      }
    }
    {
      if (true) return number;
    }
    throw new Error("Missing return statement in function");
  }

  final public double real() throws ParseException {
    int sign = 1;
    double tenFactor = 1;
    double udecimal;
    switch ((jj_ntk == -1) ? jj_ntk() : jj_ntk) {
      case PLUS:
      case MINUS:
        sign = sign();
        break;
      default:
        jj_la1[15] = jj_gen;
        ;
    }
    if (jj_2_3(2)) {
      udecimal = unsignedDecimal();
      switch ((jj_ntk == -1) ? jj_ntk() : jj_ntk) {
        case REAL_EXP:
          tenFactor = tenFactor();
          break;
        default:
          jj_la1[16] = jj_gen;
          ;
      }
    } else {
      switch ((jj_ntk == -1) ? jj_ntk() : jj_ntk) {
        case UINT:
          udecimal = unsignedInteger();
          tenFactor = tenFactor();
          break;
        default:
          jj_la1[17] = jj_gen;
          jj_consume_token(-1);
          throw new ParseException();
      }
    }
    {
      if (true) return sign * udecimal * tenFactor;
    }
    throw new Error("Missing return statement in function");
  }

  final public int sign() throws ParseException {
    switch ((jj_ntk == -1) ? jj_ntk() : jj_ntk) {
      case PLUS:
        jj_consume_token(PLUS);
      {
        if (true) return 1;
      }
      break;
      case MINUS:
        jj_consume_token(MINUS);
      {
        if (true) return -1;
      }
      break;
      default:
        jj_la1[18] = jj_gen;
        jj_consume_token(-1);
        throw new ParseException();
    }
    throw new Error("Missing return statement in function");
  }

  final public double unsignedDecimal() throws ParseException {
    int integer = 0;
    Token token;
    double fraction = 0;
    if (jj_2_4(3)) {
      switch ((jj_ntk == -1) ? jj_ntk() : jj_ntk) {
        case UINT:
          integer = unsignedInteger();
          break;
        default:
          jj_la1[19] = jj_gen;
          ;
      }
      jj_consume_token(PERIOD);
      token = jj_consume_token(UINT);
      fraction = Double.valueOf("." + token.image);
    } else {
      switch ((jj_ntk == -1) ? jj_ntk() : jj_ntk) {
        case UINT:
          integer = unsignedInteger();
          jj_consume_token(PERIOD);
          break;
        default:
          jj_la1[20] = jj_gen;
          jj_consume_token(-1);
          throw new ParseException();
      }
    }
    {
      if (true) return integer + fraction;
    }
    throw new Error("Missing return statement in function");
  }

  final public double tenFactor() throws ParseException {
    Token token;
    token = jj_consume_token(REAL_EXP);
    {
      if (true) return Double.valueOf("1" + token.image);
    }
    throw new Error("Missing return statement in function");
  }

  final public int integer() throws ParseException {
    int magnitude;
    int sign = 1;
    switch ((jj_ntk == -1) ? jj_ntk() : jj_ntk) {
      case PLUS:
      case MINUS:
        sign = sign();
        break;
      default:
        jj_la1[21] = jj_gen;
        ;
    }
    magnitude = unsignedInteger();
    {
      if (true) return sign * magnitude;
    }
    throw new Error("Missing return statement in function");
  }

  final public int unsignedInteger() throws ParseException {
    Token token;
    token = jj_consume_token(UINT);
    {
      if (true) return Integer.valueOf(token.image);
    }
    throw new Error("Missing return statement in function");
  }

  final public Unit unitIdentifier(UnitDB unitDB) throws ParseException, UnitDBException, UnitSystemException, PrefixDBException, OperationException {
    Token token;
    Unit unit;
    switch ((jj_ntk == -1) ? jj_ntk() : jj_ntk) {
      case T:
        token = jj_consume_token(T);
        break;
      case NAME:
        token = jj_consume_token(NAME);
        break;
      case SYMBOL:
        token = jj_consume_token(SYMBOL);
        break;
      default:
        jj_la1[22] = jj_gen;
        jj_consume_token(-1);
        throw new ParseException();
    }
    String string = token.image;
    double scale = 1;

    for (unit = getUnit(unitDB, string);
         unit == null;
         unit = getUnit(unitDB, string)) {
      Prefix prefix = getPrefix(string);
      if (prefix == null) {
        try {
          // System.err.println("Unknown unit: \"" + string + '"');
          unit = UnknownUnit.create(string);
          break;
        } catch (NameException e) {
        }      // shouldn't happen
      }
      scale *= prefix.getValue();
      string = string.substring(prefix.length());
    }
    unit = unit.multiplyBy(scale);
    {
      if (true) return unit;
    }
    throw new Error("Missing return statement in function");
  }

  /*
   * See <http://www.cl.cam.ac.uk/~mgk25/iso-time.html> for a discussion of the
   * relevant timestamp format or lookup "ISO 8601".
   */
  final public Date timeOriginExpr() throws ParseException {
    Calendar calendar;
    calendar = dateExpr();
    if (jj_2_6(2)) {
      switch ((jj_ntk == -1) ? jj_ntk() : jj_ntk) {
        case T:
          jj_consume_token(T);
          break;
        case SP:
          jj_consume_token(SP);
          break;
        default:
          jj_la1[23] = jj_gen;
          jj_consume_token(-1);
          throw new ParseException();
      }
      clockExpr(calendar);
      if (jj_2_5(2)) {
        switch ((jj_ntk == -1) ? jj_ntk() : jj_ntk) {
          case SP:
            jj_consume_token(SP);
            break;
          default:
            jj_la1[24] = jj_gen;
            ;
        }
        zoneExpr(calendar);
      } else {
        ;
      }
    } else {
      ;
    }
    {
      if (true) return calendar.getTime();
    }
    throw new Error("Missing return statement in function");
  }

  final public Calendar dateExpr() throws ParseException {
    int sign = 1;
    int year;
    int month = 1;
    int day = 1;
    boolean packed = true;
    switch ((jj_ntk == -1) ? jj_ntk() : jj_ntk) {
      case PLUS:
      case MINUS:
        sign = sign();
        break;
      default:
        jj_la1[25] = jj_gen;
        ;
    }
    year = unsignedInteger();
    switch ((jj_ntk == -1) ? jj_ntk() : jj_ntk) {
      case MINUS:
        jj_consume_token(MINUS);
        month = unsignedInteger();
        packed = false;
        switch ((jj_ntk == -1) ? jj_ntk() : jj_ntk) {
          case MINUS:
            jj_consume_token(MINUS);
            day = unsignedInteger();
            break;
          default:
            jj_la1[26] = jj_gen;
            ;
        }
        break;
      default:
        jj_la1[27] = jj_gen;
        ;
    }
    if (packed) {
      if (year >= 10000101) {
        day = year % 100;
        year /= 100;
      }
      if (year >= 100001) {
        month = year % 100;
        year /= 100;
      }
      if (sign < 0)
        year = -year;
    }
    if (month < 1 || month > 12) {
      if (true) throw new ParseException("invalid month in timestamp");
    }
    if (day < 1 || day > 31) {
      if (true) throw new ParseException("invalid day in timestamp");
    }
    Calendar calendar = Calendar.getInstance(TimeZone.getTimeZone("UTC"));
    calendar.clear();
    calendar.set(year, month - 1, day);
    {
      if (true) return calendar;
    }
    throw new Error("Missing return statement in function");
  }

  final public Calendar clockExpr(Calendar calendar) throws ParseException {
    double hour;
    int minute = 0;
    double seconds = 0;
    boolean packed = true;
    if (jj_2_7(2147483647)) {
      hour = unsignedDecimal();
    } else {
      switch ((jj_ntk == -1) ? jj_ntk() : jj_ntk) {
        case UINT:
          hour = unsignedInteger();
          break;
        default:
          jj_la1[28] = jj_gen;
          jj_consume_token(-1);
          throw new ParseException();
      }
    }
    switch ((jj_ntk == -1) ? jj_ntk() : jj_ntk) {
      case COLON:
        jj_consume_token(COLON);
        minute = unsignedInteger();
        packed = false;
        switch ((jj_ntk == -1) ? jj_ntk() : jj_ntk) {
          case COLON:
            jj_consume_token(COLON);
            if (jj_2_8(2147483647)) {
              seconds = unsignedDecimal();
            } else {
              switch ((jj_ntk == -1) ? jj_ntk() : jj_ntk) {
                case UINT:
                  seconds = unsignedInteger();
                  break;
                default:
                  jj_la1[29] = jj_gen;
                  jj_consume_token(-1);
                  throw new ParseException();
              }
            }
            break;
          default:
            jj_la1[30] = jj_gen;
            ;
        }
        break;
      default:
        jj_la1[31] = jj_gen;
        ;
    }
    if (packed) {
      if (hour >= 100000) {
        seconds = hour % 100;
        hour /= 100;
      }
      if (hour >= 1000) {
        minute = (int) (hour % 100);
        hour /= 100;
      }
    }
    if (hour < 0 || hour > 23) {
      if (true) throw new ParseException("invalid hour in timestamp");
    }
    if (minute < 0 || minute > 59) {
      if (true) throw new ParseException("invalid minute in timestamp");
    }
    if (seconds < 0 || seconds > 61) {
      if (true) throw new ParseException("invalid seconds in timestamp");
    }
    calendar.set(Calendar.HOUR_OF_DAY, (int) Math.round(hour));
    calendar.set(Calendar.MINUTE, minute);
    int s = (int) seconds;
    calendar.set(Calendar.SECOND, s);
    int ms = (int) ((seconds - s) * 1000);
    calendar.set(Calendar.MILLISECOND, ms);
    {
      if (true) return calendar;
    }
    throw new Error("Missing return statement in function");
  }

  final public Calendar zoneExpr(Calendar calendar) throws ParseException {
    int sign = 1;
    int zoneHour;
    int zoneMinute = 0;
    Token token;
    TimeZone timeZone;
    switch ((jj_ntk == -1) ? jj_ntk() : jj_ntk) {
      case PLUS:
      case MINUS:
      case UINT:
        switch ((jj_ntk == -1) ? jj_ntk() : jj_ntk) {
          case PLUS:
          case MINUS:
            sign = sign();
            break;
          default:
            jj_la1[32] = jj_gen;
            ;
        }
        zoneHour = unsignedInteger();
        switch ((jj_ntk == -1) ? jj_ntk() : jj_ntk) {
          case COLON:
          case UINT:
            switch ((jj_ntk == -1) ? jj_ntk() : jj_ntk) {
              case COLON:
                jj_consume_token(COLON);
                break;
              default:
                jj_la1[33] = jj_gen;
                ;
            }
            zoneMinute = unsignedInteger();
            break;
          default:
            jj_la1[34] = jj_gen;
            ;
        }
        if (zoneHour >= 100) {
          zoneMinute += zoneHour % 100;
          zoneHour /= 100;
        }
        if (zoneHour > 23 || zoneMinute > 59) {
          {
            if (true) throw new ParseException("invalid time-zone in timestamp");
          }
        }
        timeZone = TimeZone.getTimeZone("UTC");
        timeZone.setRawOffset(sign * (zoneHour * 60 + zoneMinute) * 60 * 1000);
        break;
      case NAME:
        token = jj_consume_token(NAME);
        timeZone = TimeZone.getTimeZone(token.image);
        break;
      default:
        jj_la1[35] = jj_gen;
        jj_consume_token(-1);
        throw new ParseException();
    }
    calendar.setTimeZone(timeZone);
    {
      if (true) return calendar;
    }
    throw new Error("Missing return statement in function");
  }

  private boolean jj_2_1(int xla) {
    jj_la = xla;
    jj_lastpos = jj_scanpos = token;
    try {
      return !jj_3_1();
    } catch (LookaheadSuccess ls) {
      return true;
    } finally {
      jj_save(0, xla);
    }
  }

  private boolean jj_2_2(int xla) {
    jj_la = xla;
    jj_lastpos = jj_scanpos = token;
    try {
      return !jj_3_2();
    } catch (LookaheadSuccess ls) {
      return true;
    } finally {
      jj_save(1, xla);
    }
  }

  private boolean jj_2_3(int xla) {
    jj_la = xla;
    jj_lastpos = jj_scanpos = token;
    try {
      return !jj_3_3();
    } catch (LookaheadSuccess ls) {
      return true;
    } finally {
      jj_save(2, xla);
    }
  }

  private boolean jj_2_4(int xla) {
    jj_la = xla;
    jj_lastpos = jj_scanpos = token;
    try {
      return !jj_3_4();
    } catch (LookaheadSuccess ls) {
      return true;
    } finally {
      jj_save(3, xla);
    }
  }

  private boolean jj_2_5(int xla) {
    jj_la = xla;
    jj_lastpos = jj_scanpos = token;
    try {
      return !jj_3_5();
    } catch (LookaheadSuccess ls) {
      return true;
    } finally {
      jj_save(4, xla);
    }
  }

  private boolean jj_2_6(int xla) {
    jj_la = xla;
    jj_lastpos = jj_scanpos = token;
    try {
      return !jj_3_6();
    } catch (LookaheadSuccess ls) {
      return true;
    } finally {
      jj_save(5, xla);
    }
  }

  private boolean jj_2_7(int xla) {
    jj_la = xla;
    jj_lastpos = jj_scanpos = token;
    try {
      return !jj_3_7();
    } catch (LookaheadSuccess ls) {
      return true;
    } finally {
      jj_save(6, xla);
    }
  }

  private boolean jj_2_8(int xla) {
    jj_la = xla;
    jj_lastpos = jj_scanpos = token;
    try {
      return !jj_3_8();
    } catch (LookaheadSuccess ls) {
      return true;
    } finally {
      jj_save(7, xla);
    }
  }

  private boolean jj_3R_40() {
    if (jj_scan_token(LG)) return true;
    return false;
  }

  private boolean jj_3R_33() {
    if (jj_3R_24()) return true;
    return false;
  }

  private boolean jj_3R_39() {
    if (jj_scan_token(LN)) return true;
    return false;
  }

  private boolean jj_3R_41() {
    if (jj_3R_3()) return true;
    return false;
  }

  private boolean jj_3R_23() {
    Token xsp;
    xsp = jj_scanpos;
    if (jj_3R_33()) jj_scanpos = xsp;
    if (jj_3R_14()) return true;
    return false;
  }

  private boolean jj_3R_38() {
    if (jj_scan_token(LB)) return true;
    return false;
  }

  private boolean jj_3_8() {
    if (jj_3R_5()) return true;
    return false;
  }

  private boolean jj_3_5() {
    Token xsp;
    xsp = jj_scanpos;
    if (jj_scan_token(1)) jj_scanpos = xsp;
    if (jj_3R_7()) return true;
    return false;
  }

  private boolean jj_3R_31() {
    Token xsp;
    xsp = jj_scanpos;
    if (jj_3R_38()) {
      jj_scanpos = xsp;
      if (jj_3R_39()) {
        jj_scanpos = xsp;
        if (jj_3R_40()) return true;
      }
    }
    if (jj_3R_41()) return true;
    return false;
  }

  private boolean jj_3R_26() {
    if (jj_scan_token(REAL_EXP)) return true;
    return false;
  }

  private boolean jj_3_6() {
    Token xsp;
    xsp = jj_scanpos;
    if (jj_scan_token(17)) {
      jj_scanpos = xsp;
      if (jj_scan_token(1)) return true;
    }
    if (jj_3R_8()) return true;
    return false;
  }

  private boolean jj_3R_13() {
    if (jj_3R_14()) return true;
    if (jj_scan_token(PERIOD)) return true;
    return false;
  }

  private boolean jj_3R_22() {
    if (jj_scan_token(LPAREN)) return true;
    Token xsp;
    xsp = jj_scanpos;
    if (jj_scan_token(1)) jj_scanpos = xsp;
    if (jj_3R_32()) return true;
    return false;
  }

  private boolean jj_3R_21() {
    if (jj_3R_31()) return true;
    return false;
  }

  private boolean jj_3R_20() {
    if (jj_3R_30()) return true;
    return false;
  }

  private boolean jj_3R_6() {
    if (jj_3R_14()) return true;
    return false;
  }

  private boolean jj_3_7() {
    if (jj_3R_5()) return true;
    return false;
  }

  private boolean jj_3_4() {
    Token xsp;
    xsp = jj_scanpos;
    if (jj_3R_6()) jj_scanpos = xsp;
    if (jj_scan_token(PERIOD)) return true;
    if (jj_scan_token(UINT)) return true;
    return false;
  }

  private boolean jj_3R_19() {
    if (jj_3R_29()) return true;
    return false;
  }

  private boolean jj_3R_18() {
    if (jj_3R_14()) return true;
    return false;
  }

  private boolean jj_3R_32() {
    if (jj_3R_41()) return true;
    return false;
  }

  private boolean jj_3R_17() {
    if (jj_3R_5()) return true;
    return false;
  }

  private boolean jj_3R_5() {
    Token xsp;
    xsp = jj_scanpos;
    if (jj_3_4()) {
      jj_scanpos = xsp;
      if (jj_3R_13()) return true;
    }
    return false;
  }

  private boolean jj_3R_9() {
    Token xsp;
    xsp = jj_scanpos;
    if (jj_3R_19()) {
      jj_scanpos = xsp;
      if (jj_3R_20()) {
        jj_scanpos = xsp;
        if (jj_3R_21()) {
          jj_scanpos = xsp;
          if (jj_3R_22()) return true;
        }
      }
    }
    return false;
  }

  private boolean jj_3R_35() {
    if (jj_scan_token(MINUS)) return true;
    return false;
  }

  private boolean jj_3R_8() {
    Token xsp;
    xsp = jj_scanpos;
    if (jj_3R_17()) {
      jj_scanpos = xsp;
      if (jj_3R_18()) return true;
    }
    return false;
  }

  private boolean jj_3R_34() {
    if (jj_scan_token(PLUS)) return true;
    return false;
  }

  private boolean jj_3R_24() {
    Token xsp;
    xsp = jj_scanpos;
    if (jj_3R_34()) {
      jj_scanpos = xsp;
      if (jj_3R_35()) return true;
    }
    return false;
  }

  private boolean jj_3R_12() {
    if (jj_3R_14()) return true;
    if (jj_3R_26()) return true;
    return false;
  }

  private boolean jj_3R_25() {
    if (jj_3R_26()) return true;
    return false;
  }

  private boolean jj_3R_16() {
    if (jj_scan_token(NAME)) return true;
    return false;
  }

  private boolean jj_3R_10() {
    Token xsp;
    xsp = jj_scanpos;
    if (jj_scan_token(11)) jj_scanpos = xsp;
    if (jj_3R_23()) return true;
    return false;
  }

  private boolean jj_3_3() {
    if (jj_3R_5()) return true;
    Token xsp;
    xsp = jj_scanpos;
    if (jj_3R_25()) jj_scanpos = xsp;
    return false;
  }

  private boolean jj_3R_3() {
    if (jj_3R_9()) return true;
    Token xsp;
    xsp = jj_scanpos;
    if (jj_3R_10()) jj_scanpos = xsp;
    return false;
  }

  private boolean jj_3R_11() {
    if (jj_3R_24()) return true;
    return false;
  }

  private boolean jj_3R_4() {
    Token xsp;
    xsp = jj_scanpos;
    if (jj_3R_11()) jj_scanpos = xsp;
    xsp = jj_scanpos;
    if (jj_3_3()) {
      jj_scanpos = xsp;
      if (jj_3R_12()) return true;
    }
    return false;
  }

  private boolean jj_3_2() {
    if (jj_3R_4()) return true;
    return false;
  }

  private boolean jj_3R_30() {
    Token xsp;
    xsp = jj_scanpos;
    if (jj_scan_token(17)) {
      jj_scanpos = xsp;
      if (jj_scan_token(18)) {
        jj_scanpos = xsp;
        if (jj_scan_token(16)) return true;
      }
    }
    return false;
  }

  private boolean jj_3R_28() {
    Token xsp;
    xsp = jj_scanpos;
    if (jj_scan_token(4)) jj_scanpos = xsp;
    if (jj_3R_14()) return true;
    return false;
  }

  private boolean jj_3R_27() {
    if (jj_3R_24()) return true;
    return false;
  }

  private boolean jj_3R_15() {
    Token xsp;
    xsp = jj_scanpos;
    if (jj_3R_27()) jj_scanpos = xsp;
    if (jj_3R_14()) return true;
    xsp = jj_scanpos;
    if (jj_3R_28()) jj_scanpos = xsp;
    return false;
  }

  private boolean jj_3R_37() {
    if (jj_3R_23()) return true;
    return false;
  }

  private boolean jj_3R_2() {
    Token xsp;
    xsp = jj_scanpos;
    if (jj_scan_token(12)) {
      jj_scanpos = xsp;
      if (jj_scan_token(13)) {
        jj_scanpos = xsp;
        if (jj_scan_token(1)) return true;
      }
    }
    return false;
  }

  private boolean jj_3R_36() {
    if (jj_3R_4()) return true;
    return false;
  }

  private boolean jj_3_1() {
    Token xsp;
    xsp = jj_scanpos;
    if (jj_3R_2()) jj_scanpos = xsp;
    if (jj_3R_3()) return true;
    return false;
  }

  private boolean jj_3R_7() {
    Token xsp;
    xsp = jj_scanpos;
    if (jj_3R_15()) {
      jj_scanpos = xsp;
      if (jj_3R_16()) return true;
    }
    return false;
  }

  private boolean jj_3R_14() {
    if (jj_scan_token(UINT)) return true;
    return false;
  }

  private boolean jj_3R_29() {
    Token xsp;
    xsp = jj_scanpos;
    if (jj_3R_36()) {
      jj_scanpos = xsp;
      if (jj_3R_37()) return true;
    }
    return false;
  }

  /**
   * Generated Token Manager.
   */
  public StandardUnitFormatTokenManager token_source;
  SimpleCharStream jj_input_stream;
  /**
   * Current token.
   */
  public Token token;
  /**
   * Next token.
   */
  public Token jj_nt;
  private int jj_ntk;
  private Token jj_scanpos, jj_lastpos;
  private int jj_la;
  private int jj_gen;
  final private int[] jj_la1 = new int[36];
  static private int[] jj_la1_0;

  static {
    jj_la1_init_0();
  }

  private static void jj_la1_init_0() {
    jj_la1_0 = new int[]{0x3f112c, 0x102c, 0x8000, 0x3f712e, 0x3002, 0x3002, 0x4000, 0x800, 0x82c, 0x2, 0x2, 0x3f112c, 0x380000, 0x2, 0x2c, 0xc, 0x400, 0x20, 0xc, 0x20, 0x20, 0xc, 0x70000, 0x20002, 0x2, 0xc, 0x8, 0x8, 0x20, 0x20, 0x10, 0x10, 0xc, 0x10, 0x30, 0x4002c,};
  }

  final private JJCalls[] jj_2_rtns = new JJCalls[8];
  private boolean jj_rescan = false;
  private int jj_gc = 0;

  /**
   * Constructor with InputStream.
   */
  public StandardUnitFormat(java.io.InputStream stream) {
    this(stream, null);
  }

  /**
   * Constructor with InputStream and supplied encoding
   */
  public StandardUnitFormat(java.io.InputStream stream, String encoding) {
    try {
      jj_input_stream = new SimpleCharStream(stream, encoding, 1, 1);
    } catch (java.io.UnsupportedEncodingException e) {
      throw new RuntimeException(e);
    }
    token_source = new StandardUnitFormatTokenManager(jj_input_stream);
    token = new Token();
    jj_ntk = -1;
    jj_gen = 0;
    for (int i = 0; i < 36; i++) jj_la1[i] = -1;
    for (int i = 0; i < jj_2_rtns.length; i++) jj_2_rtns[i] = new JJCalls();
  }

  /**
   * Reinitialise.
   */
  public void ReInit(java.io.InputStream stream) {
    ReInit(stream, null);
  }

  /**
   * Reinitialise.
   */
  public void ReInit(java.io.InputStream stream, String encoding) {
    try {
      jj_input_stream.ReInit(stream, encoding, 1, 1);
    } catch (java.io.UnsupportedEncodingException e) {
      throw new RuntimeException(e);
    }
    token_source.ReInit(jj_input_stream);
    token = new Token();
    jj_ntk = -1;
    jj_gen = 0;
    for (int i = 0; i < 36; i++) jj_la1[i] = -1;
    for (int i = 0; i < jj_2_rtns.length; i++) jj_2_rtns[i] = new JJCalls();
  }

  /**
   * Constructor.
   */
  public StandardUnitFormat(java.io.Reader stream) {
    jj_input_stream = new SimpleCharStream(stream, 1, 1);
    token_source = new StandardUnitFormatTokenManager(jj_input_stream);
    token = new Token();
    jj_ntk = -1;
    jj_gen = 0;
    for (int i = 0; i < 36; i++) jj_la1[i] = -1;
    for (int i = 0; i < jj_2_rtns.length; i++) jj_2_rtns[i] = new JJCalls();
  }

  /**
   * Reinitialise.
   */
  public void ReInit(java.io.Reader stream) {
    jj_input_stream.ReInit(stream, 1, 1);
    token_source.ReInit(jj_input_stream);
    token = new Token();
    jj_ntk = -1;
    jj_gen = 0;
    for (int i = 0; i < 36; i++) jj_la1[i] = -1;
    for (int i = 0; i < jj_2_rtns.length; i++) jj_2_rtns[i] = new JJCalls();
  }

  /**
   * Constructor with generated Token Manager.
   */
  public StandardUnitFormat(StandardUnitFormatTokenManager tm) {
    token_source = tm;
    token = new Token();
    jj_ntk = -1;
    jj_gen = 0;
    for (int i = 0; i < 36; i++) jj_la1[i] = -1;
    for (int i = 0; i < jj_2_rtns.length; i++) jj_2_rtns[i] = new JJCalls();
  }

  /**
   * Reinitialise.
   */
  public void ReInit(StandardUnitFormatTokenManager tm) {
    token_source = tm;
    token = new Token();
    jj_ntk = -1;
    jj_gen = 0;
    for (int i = 0; i < 36; i++) jj_la1[i] = -1;
    for (int i = 0; i < jj_2_rtns.length; i++) jj_2_rtns[i] = new JJCalls();
  }

  private Token jj_consume_token(int kind) throws ParseException {
    Token oldToken;
    if ((oldToken = token).next != null) token = token.next;
    else token = token.next = token_source.getNextToken();
    jj_ntk = -1;
    if (token.kind == kind) {
      jj_gen++;
      if (++jj_gc > 100) {
        jj_gc = 0;
        for (int i = 0; i < jj_2_rtns.length; i++) {
          JJCalls c = jj_2_rtns[i];
          while (c != null) {
            if (c.gen < jj_gen) c.first = null;
            c = c.next;
          }
        }
      }
      return token;
    }
    token = oldToken;
    jj_kind = kind;
    throw generateParseException();
  }

  static private final class LookaheadSuccess extends java.lang.Error {
  }

  final private LookaheadSuccess jj_ls = new LookaheadSuccess();

  private boolean jj_scan_token(int kind) {
    if (jj_scanpos == jj_lastpos) {
      jj_la--;
      if (jj_scanpos.next == null) {
        jj_lastpos = jj_scanpos = jj_scanpos.next = token_source.getNextToken();
      } else {
        jj_lastpos = jj_scanpos = jj_scanpos.next;
      }
    } else {
      jj_scanpos = jj_scanpos.next;
    }
    if (jj_rescan) {
      int i = 0;
      Token tok = token;
      while (tok != null && tok != jj_scanpos) {
        i++;
        tok = tok.next;
      }
      if (tok != null) jj_add_error_token(kind, i);
    }
    if (jj_scanpos.kind != kind) return true;
    if (jj_la == 0 && jj_scanpos == jj_lastpos) throw jj_ls;
    return false;
  }


  /**
   * Get the next Token.
   */
  final public Token getNextToken() {
    if (token.next != null) token = token.next;
    else token = token.next = token_source.getNextToken();
    jj_ntk = -1;
    jj_gen++;
    return token;
  }

  /**
   * Get the specific Token.
   */
  final public Token getToken(int index) {
    Token t = token;
    for (int i = 0; i < index; i++) {
      if (t.next != null) t = t.next;
      else t = t.next = token_source.getNextToken();
    }
    return t;
  }

  private int jj_ntk() {
    if ((jj_nt = token.next) == null)
      return (jj_ntk = (token.next = token_source.getNextToken()).kind);
    else
      return (jj_ntk = jj_nt.kind);
  }

  private java.util.List<int[]> jj_expentries = new java.util.ArrayList<int[]>();
  private int[] jj_expentry;
  private int jj_kind = -1;
  private int[] jj_lasttokens = new int[100];
  private int jj_endpos;

  private void jj_add_error_token(int kind, int pos) {
    if (pos >= 100) return;
    if (pos == jj_endpos + 1) {
      jj_lasttokens[jj_endpos++] = kind;
    } else if (jj_endpos != 0) {
      jj_expentry = new int[jj_endpos];
      for (int i = 0; i < jj_endpos; i++) {
        jj_expentry[i] = jj_lasttokens[i];
      }
      jj_entries_loop:
      for (java.util.Iterator it = jj_expentries.iterator(); it.hasNext(); ) {
        int[] oldentry = (int[]) (it.next());
        if (oldentry.length == jj_expentry.length) {
          for (int i = 0; i < jj_expentry.length; i++) {
            if (oldentry[i] != jj_expentry[i]) {
              continue jj_entries_loop;
            }
          }
          jj_expentries.add(jj_expentry);
          break jj_entries_loop;
        }
      }
      if (pos != 0) jj_lasttokens[(jj_endpos = pos) - 1] = kind;
    }
  }

  /**
   * Generate ParseException.
   */
  public ParseException generateParseException() {
    jj_expentries.clear();
    boolean[] la1tokens = new boolean[22];
    if (jj_kind >= 0) {
      la1tokens[jj_kind] = true;
      jj_kind = -1;
    }
    for (int i = 0; i < 36; i++) {
      if (jj_la1[i] == jj_gen) {
        for (int j = 0; j < 32; j++) {
          if ((jj_la1_0[i] & (1 << j)) != 0) {
            la1tokens[j] = true;
          }
        }
      }
    }
    for (int i = 0; i < 22; i++) {
      if (la1tokens[i]) {
        jj_expentry = new int[1];
        jj_expentry[0] = i;
        jj_expentries.add(jj_expentry);
      }
    }
    jj_endpos = 0;
    jj_rescan_token();
    jj_add_error_token(0, 0);
    int[][] exptokseq = new int[jj_expentries.size()][];
    for (int i = 0; i < jj_expentries.size(); i++) {
      exptokseq[i] = jj_expentries.get(i);
    }
    return new ParseException(token, exptokseq, tokenImage);
  }

  /**
   * Enable tracing.
   */
  final public void enable_tracing() {
  }

  /**
   * Disable tracing.
   */
  final public void disable_tracing() {
  }

  private void jj_rescan_token() {
    jj_rescan = true;
    for (int i = 0; i < 8; i++) {
      try {
        JJCalls p = jj_2_rtns[i];
        do {
          if (p.gen > jj_gen) {
            jj_la = p.arg;
            jj_lastpos = jj_scanpos = p.first;
            switch (i) {
              case 0:
                jj_3_1();
                break;
              case 1:
                jj_3_2();
                break;
              case 2:
                jj_3_3();
                break;
              case 3:
                jj_3_4();
                break;
              case 4:
                jj_3_5();
                break;
              case 5:
                jj_3_6();
                break;
              case 6:
                jj_3_7();
                break;
              case 7:
                jj_3_8();
                break;
            }
          }
          p = p.next;
        } while (p != null);
      } catch (LookaheadSuccess ls) {
      }
    }
    jj_rescan = false;
  }

  private void jj_save(int index, int xla) {
    JJCalls p = jj_2_rtns[index];
    while (p.gen > jj_gen) {
      if (p.next == null) {
        p = p.next = new JJCalls();
        break;
      }
      p = p.next;
    }
    p.gen = jj_gen + xla - jj_la;
    p.first = token;
    p.arg = xla;
  }

  static final class JJCalls {
    int gen;
    Token first;
    int arg;
    JJCalls next;
  }

}
