/*
 * Copyright (c) 1998-2018 University Corporation for Atmospheric Research/Unidata
 * See LICENSE for license information.
 */
package ucar.nc2.ui.grid;

import thredds.client.catalog.ServiceType;
import thredds.client.catalog.Dataset;
import thredds.client.catalog.tools.DataFactory;
import ucar.nc2.constants.FeatureType;
import ucar.nc2.dataset.CoordinateAxis1D;
import ucar.nc2.dataset.CoordinateAxis1DTime;
import ucar.nc2.dataset.NetcdfDataset;
import ucar.nc2.dt.GridCoordSystem;
import ucar.nc2.dt.GridDataset;
import ucar.nc2.dt.GridDatatype;
import ucar.nc2.ui.geoloc.NavigatedPanel;
import ucar.nc2.ui.geoloc.ProjectionManager;
import ucar.nc2.ui.gis.MapBean;
import ucar.nc2.ui.util.NamedObjects;
import ucar.nc2.ui.util.Renderer;
import ucar.ui.widget.*;
import ucar.ui.widget.PopupMenu;
import ucar.ui.widget.ProgressMonitor;
import ucar.ui.util.NamedObject;
import ucar.unidata.geoloc.Projection;
import ucar.util.prefs.PreferencesExt;
import javax.swing.*;
import javax.swing.border.EtchedBorder;
import java.awt.*;
import java.awt.event.ActionEvent;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Formatter;
import java.util.List;

/** This is the thredds Data Viewer application User Interface for Grids. */
public class GridUI extends JPanel {
  private static final String GEOTIFF_FILECHOOSER_DEFAULTDIR = "geotiffDefDir";
  // private TopLevel topLevel;
  private final PreferencesExt store;
  private final FileManager fileChooser;

  // Package private access
  SuperComboBox fieldChooser, levelChooser, timeChooser, ensembleChooser, runtimeChooser;
  NavigatedPanel panz = new NavigatedPanel();
  VertPanel vertPanel;
  GridTable gridTable;
  JLabel dataValueLabel, positionLabel;

  // the main components
  private GridController controller;
  private ColorScale.Panel colorScalePanel;

  private GeoGridTable dsTable;

  // UI components that need global scope
  private TextHistoryPane datasetInfoTA, ncmlTA;
  private JPanel drawingPanel;
  private JSplitPane splitDraw;
  private JComboBox csDataMinMax;
  private PopupMenu mapBeanMenu;

  private JLabel datasetNameLabel;

  // the various managers and dialog boxes
  private ProjectionManager projManager;
  private IndependentWindow infoWindow;
  private IndependentWindow ncmlWindow;
  private IndependentWindow gtWindow;
  private JDialog dsDialog;
  private FileManager geotiffFileChooser;

  // toolbars
  private JPanel fieldPanel, toolPanel;
  private JToolBar navToolbar, moveToolbar;
  private AbstractAction navToolbarAction, moveToolbarAction;

  // actions
  private AbstractAction redrawAction;
  private AbstractAction showDatasetInfoAction;
  private AbstractAction showNcMLAction;
  private AbstractAction showGridTableAction;
  private AbstractAction showGridDatasetInfoAction;
  private AbstractAction showNetcdfDatasetAction;
  private AbstractAction minmaxHorizAction, minmaxLogAction, minmaxHoldAction;
  private AbstractAction fieldLoopAction, levelLoopAction, timeLoopAction;
  private AbstractAction chooseProjectionAction, saveCurrentProjectionAction;

  // state
  private boolean selected;
  private int mapBeanCount;

  // debugging


  public GridUI(PreferencesExt pstore, RootPaneContainer root, FileManager fileChooser, int defaultHeight) {
    // this.topUI = topUI;
    this.store = pstore;
    this.fileChooser = fileChooser;

    try {
      choosers = new ArrayList<>();
      fieldChooser = new SuperComboBox(root, "field", true, null);
      choosers.add(new Chooser("field", fieldChooser, true));
      runtimeChooser = new SuperComboBox(root, "runtime", false, null);
      choosers.add(new Chooser("runtime", runtimeChooser, false));
      timeChooser = new SuperComboBox(root, "time", false, null);
      choosers.add(new Chooser("time", timeChooser, false));
      ensembleChooser = new SuperComboBox(root, "ensemble", false, null);
      choosers.add(new Chooser("ensemble", ensembleChooser, false));
      levelChooser = new SuperComboBox(root, "level", false, null);
      choosers.add(new Chooser("level", levelChooser, false));

      makeActionsDataset();
      makeActionsToolbars();

      gridTable = new GridTable("field");
      gtWindow = new IndependentWindow("Grid Table Information", BAMutil.getImage("nj22/GDVs"), gridTable.getPanel());

      PreferencesExt dsNode = (PreferencesExt) pstore.node("DatasetTable");
      dsTable = new GeoGridTable(dsNode, true);
      dsDialog = dsTable.makeDialog(root, "NetcdfDataset Info", false);
      // dsDialog.setIconImage( BAMutil.getImage( "GDVs"));
      Rectangle bounds = (Rectangle) dsNode.getBean("DialogBounds", new Rectangle(50, 50, 800, 450));
      dsDialog.setBounds(bounds);

      controller = new GridController(this, store);
      makeUI(defaultHeight);
      controller.finishInit();

      // other components
      geotiffFileChooser = new FileManager(null);
      geotiffFileChooser.setCurrentDirectory(store.get(GEOTIFF_FILECHOOSER_DEFAULTDIR, "."));

    } catch (Exception e) {
      System.out.println("UI creation failed");
      e.printStackTrace();
    }
  }

  /** save all data in the PersistentStore */
  public void storePersistentData() {
    store.putInt("vertSplit", splitDraw.getDividerLocation());

    store.putBoolean("navToolbarAction", (Boolean) navToolbarAction.getValue(BAMutil.STATE));
    store.putBoolean("moveToolbarAction", (Boolean) moveToolbarAction.getValue(BAMutil.STATE));

    if (projManager != null)
      projManager.storePersistentData();

    dsTable.save();
    dsTable.getPrefs().putBeanObject("DialogBounds", dsDialog.getBounds());

    store.put(GEOTIFF_FILECHOOSER_DEFAULTDIR, geotiffFileChooser.getCurrentDirectory());

    controller.storePersistentData();
  }

  boolean isSelected() {
    return selected;
  }

  void setSelected(boolean b) {
    selected = b;

    showGridTableAction.setEnabled(b);
    showNcMLAction.setEnabled(b);
    showNcMLAction.setEnabled(b);
    showNetcdfDatasetAction.setEnabled(b);
    showGridDatasetInfoAction.setEnabled(b);

    navToolbarAction.setEnabled(b);
    moveToolbarAction.setEnabled(b);

    controller.showGridAction.setEnabled(b);
    controller.showContoursAction.setEnabled(b);
    controller.showContourLabelsAction.setEnabled(b);
    redrawAction.setEnabled(b);

    minmaxHorizAction.setEnabled(b);
    minmaxLogAction.setEnabled(b);
    minmaxHoldAction.setEnabled(b);

    fieldLoopAction.setEnabled(b);
    levelLoopAction.setEnabled(b);
    timeLoopAction.setEnabled(b);

    panz.setEnabledActions(b);
  }

  // add a MapBean to the User Interface
  public void addMapBean(MapBean mb) {
    mapBeanMenu.addAction(mb.getActionDesc(), mb.getIcon(), mb.getAction());

    // first one is the "default"
    if (mapBeanCount == 0) {
      setMapRenderer(mb.getRenderer());
    }
    mapBeanCount++;

    mb.addPropertyChangeListener(e -> {
      if (e.getPropertyName().equals("Renderer")) {
        setMapRenderer((Renderer) e.getNewValue());
      }
    });
  }

  void setMapRenderer(Renderer mapRenderer) {
    controller.setMapRenderer(mapRenderer);
  }

  public void clear() {
    controller.clear();
    gridTable.clear();
  }

  public void setDataset(Dataset ds) {
    if (ds == null)
      return;

    OpenDatasetTask openTask = new OpenDatasetTask(ds);
    ProgressMonitor pm = new ProgressMonitor(openTask, (e) -> {
      controller.showDataset();
      gridTable.setDataset(controller.getFields());
      datasetNameLabel.setText("Dataset:  " + controller.getDatasetUrlString());
      setSelected(true);
      gtWindow.hide();
    });
    pm.start(this, "Open Dataset " + ds.getName(), 100);
  }


  public void setDataset(GridDataset ds) {
    controller.setGridDataset(ds);
    controller.showDataset();
    datasetNameLabel.setText("Dataset:  " + controller.getDatasetUrlString());
    gridTable.setDataset(controller.getFields());
  }

  void setFields(java.util.List<GridDatatype> fields) {
    fieldChooser.setCollection(fields.iterator());
  }

  void setField(GridDatatype field) {
    fieldChooser.setToolTipText(field.getName());

    GridCoordSystem gcs = field.getCoordinateSystem();

    // levels
    CoordinateAxis1D axis = gcs.getVerticalAxis();
    setChooserWanted("level", axis != null);
    if (axis != null) {
      List<NamedObject> levels = NamedObjects.getNames(axis);
      levelChooser.setCollection(levels.iterator(), true);
      NamedObject no = levels.get(controller.getCurrentLevelIndex());
      levelChooser.setSelectedByName(no.getName());
    }

    // times
    if (gcs.hasTimeAxis()) {
      axis = gcs.hasTimeAxis1D() ? gcs.getTimeAxis1D() : gcs.getTimeAxisForRun(0);
      setChooserWanted("time", axis != null);
      if (axis != null) {
        List<NamedObject> names = NamedObjects.getNames((CoordinateAxis1DTime) axis);
        timeChooser.setCollection(names.iterator(), true);
        NamedObject no = names.get(controller.getCurrentTimeIndex());
        timeChooser.setSelectedByName(no.getName());
      }
    } else {
      setChooserWanted("time", false);
    }

    axis = gcs.getEnsembleAxis();
    setChooserWanted("ensemble", axis != null);
    if (axis != null) {
      List<NamedObject> names = NamedObjects.getNames(axis);
      ensembleChooser.setCollection(names.iterator(), true);
      NamedObject no = names.get(controller.getCurrentEnsembleIndex());
      ensembleChooser.setSelectedByName(no.getName());
    }

    axis = gcs.getRunTimeAxis();
    setChooserWanted("runtime", axis != null);
    if (axis != null) {
      List<NamedObject> names = NamedObjects.getNames((CoordinateAxis1DTime) axis);
      runtimeChooser.setCollection(names.iterator(), true);
      NamedObject no = names.get(controller.getCurrentRunTimeIndex());
      runtimeChooser.setSelectedByName(no.getName());
    }

    setChoosers();

    colorScalePanel.setUnitString(field.getUnitsString());
  }

  void setDrawHorizAndVert(boolean drawHoriz, boolean drawVert) {
    drawingPanel.removeAll();
    if (drawHoriz && drawVert) {
      splitDraw.setTopComponent(panz);
      splitDraw.setBottomComponent(vertPanel);
      drawingPanel.add(splitDraw, BorderLayout.CENTER);
    } else if (drawHoriz) {
      drawingPanel.add(panz, BorderLayout.CENTER);
    } else if (drawVert) {
      drawingPanel.add(splitDraw, BorderLayout.CENTER);
    }
  }

  // actions that control the dataset
  private void makeActionsDataset() {

    // choose local dataset
    AbstractAction chooseLocalDatasetAction = new AbstractAction() {
      public void actionPerformed(ActionEvent e) {
        String filename = fileChooser.chooseFilename();
        if (filename == null)
          return;

        Dataset invDs;
        try {
          invDs = Dataset.makeStandalone(filename, FeatureType.GRID.toString(), "", ServiceType.File.toString());
        } catch (Exception ue) {
          JOptionPane.showMessageDialog(GridUI.this, "Invalid filename = <" + filename + ">\n" + ue.getMessage());
          ue.printStackTrace();
          return;
        }
        setDataset(invDs);
      }
    };
    BAMutil.setActionProperties(chooseLocalDatasetAction, "FileChooser", "open Local dataset...", false, 'L', -1);

    /*
     * saveDatasetAction = new AbstractAction() {
     * public void actionPerformed(ActionEvent e) {
     * String fname = controller.getDatasetName();
     * if (fname != null) {
     * savedDatasetList.add( fname);
     * BAMutil.addActionToMenu( savedDatasetMenu, new DatasetAction( fname), 0);
     * }
     * }
     * };
     * BAMutil.setActionProperties( saveDatasetAction, null, "save dataset", false, 'S', 0);
     */

    // Configure
    chooseProjectionAction = new AbstractAction() {
      public void actionPerformed(ActionEvent e) {
        getProjectionManager().setVisible();
      }
    };
    BAMutil.setActionProperties(chooseProjectionAction, null, "Projection Manager...", false, 'P', 0);

    saveCurrentProjectionAction = new AbstractAction() {
      public void actionPerformed(ActionEvent e) {
        getProjectionManager();
        // set the bounding box
        // Projection proj = panz.getProjectionImpl().constructCopy();
        // proj.setDefaultMapArea(panz.getMapArea());
        // if (debug) System.out.println(" GV save projection "+ proj);

        // projManage.setMap(renderAll.get("Map")); LOOK!
        // projManager.saveProjection( proj);
      }
    };
    BAMutil.setActionProperties(saveCurrentProjectionAction, null, "save Current Projection", false, 'S', 0);

    /*
     * chooseColorScaleAction = new AbstractAction() {
     * public void actionPerformed(ActionEvent e) {
     * if (null == csManager) // lazy instantiation
     * makeColorScaleManager();
     * csManager.show();
     * }
     * };
     * BAMutil.setActionProperties( chooseColorScaleAction, null, "ColorScale Manager...", false, 'C', 0);
     * 
     */
    // redraw
    redrawAction = new AbstractAction() {
      public void actionPerformed(ActionEvent e) {
        repaint();
        controller.start(true);
        controller.draw(true);
      }
    };
    BAMutil.setActionProperties(redrawAction, "alien", "RedRaw", false, 'W', 0);

    showDatasetInfoAction = new AbstractAction() {
      public void actionPerformed(ActionEvent e) {
        if (infoWindow == null) {
          datasetInfoTA = new TextHistoryPane();
          infoWindow = new IndependentWindow("Dataset Information", BAMutil.getImage("nj22/GDVs"), datasetInfoTA);
          infoWindow.setSize(700, 700);
          infoWindow.setLocation(100, 100);
        }

        datasetInfoTA.clear();
        datasetInfoTA.appendLine(controller.getDatasetInfo());
        datasetInfoTA.gotoTop();
        infoWindow.show();
      }
    };
    BAMutil.setActionProperties(showDatasetInfoAction, "Information", "Show info...", false, 'S', -1);

    showNcMLAction = new AbstractAction() {
      public void actionPerformed(ActionEvent e) {
        if (ncmlWindow == null) {
          ncmlTA = new TextHistoryPane();
          ncmlWindow = new IndependentWindow("Dataset NcML", BAMutil.getImage("nj22/GDVs"), ncmlTA);
          ncmlWindow.setSize(700, 700);
          ncmlWindow.setLocation(200, 70);
        }

        ncmlTA.clear();
        // datasetInfoTA.appendLine( "GeoGrid XML for "+ controller.getDatasetName()+"\n");
        ncmlTA.appendLine(controller.getNcML());
        ncmlTA.gotoTop();
        ncmlWindow.show();
      }
    };
    BAMutil.setActionProperties(showNcMLAction, null, "Show NcML...", false, 'X', -1);

    showGridDatasetInfoAction = new AbstractAction() {
      public void actionPerformed(ActionEvent e) {
        if (ncmlWindow == null) {
          ncmlTA = new TextHistoryPane();
          ncmlWindow = new IndependentWindow("Dataset NcML", BAMutil.getImage("nj22/GDVs"), ncmlTA);
          ncmlWindow.setSize(700, 700);
          ncmlWindow.setLocation(200, 70);
        }

        ncmlTA.clear();
        // datasetInfoTA.appendLine( "GeoGrid XML for "+ controller.getDatasetName()+"\n");
        ncmlTA.appendLine(controller.getDatasetXML());
        ncmlTA.gotoTop();
        ncmlWindow.show();
      }
    };
    BAMutil.setActionProperties(showGridDatasetInfoAction, null, "Show GridDataset Info XML...", false, 'X', -1);

    // show gridTable
    showGridTableAction = new AbstractAction() {
      public void actionPerformed(ActionEvent e) {
        gtWindow.show();
      }
    };
    BAMutil.setActionProperties(showGridTableAction, "Table", "grid Table...", false, 'T', -1);

    // show netcdf dataset Table
    showNetcdfDatasetAction = new AbstractAction() {
      public void actionPerformed(ActionEvent e) {
        NetcdfDataset netcdfDataset = controller.getNetcdfDataset();
        if (null != netcdfDataset) {
          try {
            dsTable.setDataset(netcdfDataset, null);
          } catch (IOException e1) {
            e1.printStackTrace();
            return;
          }
          dsDialog.show();
        }
      }
    };
    BAMutil.setActionProperties(showNetcdfDatasetAction, "nj22/Netcdf", "NetcdfDataset Table Info...", false, 'D', -1);

    minmaxHorizAction = new AbstractAction() {
      public void actionPerformed(ActionEvent e) {
        csDataMinMax.setSelectedItem(ColorScale.MinMaxType.horiz);
        controller.setDataMinMaxType(ColorScale.MinMaxType.horiz);
      }
    };
    BAMutil.setActionProperties(minmaxHorizAction, null, "Horizontal plane", false, 'H', 0);

    minmaxLogAction = new AbstractAction() {
      public void actionPerformed(ActionEvent e) {
        csDataMinMax.setSelectedItem(ColorScale.MinMaxType.log);
        controller.setDataMinMaxType(ColorScale.MinMaxType.log);
      }
    };
    BAMutil.setActionProperties(minmaxLogAction, null, "log horiz plane", false, 'V', 0);

    minmaxHoldAction = new AbstractAction() {
      public void actionPerformed(ActionEvent e) {
        csDataMinMax.setSelectedItem(ColorScale.MinMaxType.hold);
        controller.setDataMinMaxType(ColorScale.MinMaxType.hold);
      }
    };
    BAMutil.setActionProperties(minmaxHoldAction, null, "Hold scale constant", false, 'C', 0);

    fieldLoopAction = new LoopControlAction(fieldChooser);
    levelLoopAction = new LoopControlAction(levelChooser);
    timeLoopAction = new LoopControlAction(timeChooser);
  }

  private void makeActionsToolbars() {
    navToolbarAction = new AbstractAction() {
      public void actionPerformed(ActionEvent e) {
        Boolean state = (Boolean) getValue(BAMutil.STATE);
        if (state)
          toolPanel.add(navToolbar);
        else
          toolPanel.remove(navToolbar);
      }
    };
    BAMutil.setActionProperties(navToolbarAction, "MagnifyPlus", "show Navigate toolbar", true, 'M', 0);
    navToolbarAction.putValue(BAMutil.STATE, store.getBoolean("navToolbarAction", true));

    moveToolbarAction = new AbstractAction() {
      public void actionPerformed(ActionEvent e) {
        Boolean state = (Boolean) getValue(BAMutil.STATE);
        if (state)
          toolPanel.add(moveToolbar);
        else
          toolPanel.remove(moveToolbar);
      }
    };
    BAMutil.setActionProperties(moveToolbarAction, "Up", "show Move toolbar", true, 'M', 0);
    moveToolbarAction.putValue(BAMutil.STATE, store.getBoolean("moveToolbarAction", true));
  }

  /*
   * private void makeSysConfigWindow() {
   * sysConfigDialog = new ucar.unidata.ui.PropertyDialog(topLevel.getRootPaneContainer(), true,
   * "System Configuration", store, "HelpDir"); // LOOK KLUDGE
   * sysConfigDialog.pack();
   * sysConfigDialog.setSize(500,200);
   * sysConfigDialog.setLocation(300,300);
   * }
   * 
   * private void makeColorScaleManager() {
   * csManager = new ColorScaleManager(topLevel.getRootPaneContainer(), store);
   * csManager.addPropertyChangeListener( new java.beans.PropertyChangeListener() {
   * public void propertyChange( java.beans.PropertyChangeEvent e) {
   * if (e.getPropertyName().equals("ColorScale")) {
   * ColorScale cs = (ColorScale) e.getNewValue();
   * cs = (ColorScale) cs.clone();
   * //System.out.println("UI: new Colorscale got "+cs);
   * colorScalePanel.setColorScale(cs);
   * controller.setColorScale(cs);
   * }
   * }
   * });
   * }
   */

  public ProjectionManager getProjectionManager() {
    if (null != projManager)
      return projManager;

    projManager = new ProjectionManager(null, store);
    projManager.addPropertyChangeListener(new java.beans.PropertyChangeListener() {
      public void propertyChange(java.beans.PropertyChangeEvent e) {
        if (e.getPropertyName().equals("ProjectionImpl")) {
          Projection p = (Projection) e.getNewValue();
          // p = p.constructCopy();
          // System.out.println("UI: new Projection "+p);
          controller.setProjection(p);
        }
      }
    });

    return projManager;
  }


  private void makeUI(int defaultHeight) {

    datasetNameLabel = new JLabel();
    /*
     * gridPP = new PrefPanel("GridView", (PreferencesExt) store.node("GridViewPrefs"));
     * gridUrlIF = gridPP.addTextComboField("url", "Gridded Data URL", null, 10, false);
     * gridPP.addButton( BAMutil.makeButtconFromAction( chooseLocalDatasetAction ));
     * gridPP.finish(true, BorderLayout.EAST);
     * gridPP.addActionListener(new ActionListener() {
     * public void actionPerformed(ActionEvent e) {
     * InvDatasetImpl ds = new InvDatasetImpl( gridUrlIF.getText(), thredds.catalog.DataType.GRID, ServiceType.NETCDF);
     * setDataset( ds);
     * }
     * });
     */

    // top tool panel
    toolPanel = new JPanel();
    toolPanel.setBorder(new EtchedBorder());
    toolPanel.setLayout(new MFlowLayout(FlowLayout.LEFT, 0, 0));

    // menus
    JMenu dataMenu = new JMenu("Dataset");
    dataMenu.setMnemonic('D');
    JMenu configMenu = new JMenu("Configure");
    configMenu.setMnemonic('C');
    JMenu toolMenu = new JMenu("Controls");
    toolMenu.setMnemonic('T');
    addActionsToMenus(dataMenu, configMenu, toolMenu);
    JMenuBar menuBar = new JMenuBar();
    menuBar.add(dataMenu);
    menuBar.add(configMenu);
    menuBar.add(toolMenu);
    toolPanel.add(menuBar);

    // field choosers
    fieldPanel = new JPanel();
    fieldPanel.setLayout(new FlowLayout(FlowLayout.LEFT, 0, 0));
    toolPanel.add(fieldPanel);

    // stride
    toolPanel.add(controller.strideSpinner);

    // buttcons
    BAMutil.addActionToContainer(toolPanel, controller.drawHorizAction);
    BAMutil.addActionToContainer(toolPanel, controller.drawVertAction);
    mapBeanMenu = MapBean.makeMapSelectButton();
    toolPanel.add(mapBeanMenu.getParentComponent());

    // the Navigated panel and its toolbars
    panz.setLayout(new FlowLayout());
    navToolbar = panz.getNavToolBar();
    moveToolbar = panz.getMoveToolBar();
    if ((Boolean) navToolbarAction.getValue(BAMutil.STATE))
      toolPanel.add(navToolbar);
    if ((Boolean) moveToolbarAction.getValue(BAMutil.STATE))
      toolPanel.add(moveToolbar);

    BAMutil.addActionToContainer(toolPanel, panz.setReferenceAction);
    BAMutil.addActionToContainer(toolPanel, controller.dataProjectionAction);
    BAMutil.addActionToContainer(toolPanel, controller.showGridAction);
    BAMutil.addActionToContainer(toolPanel, controller.showContoursAction);
    BAMutil.addActionToContainer(toolPanel, controller.showContourLabelsAction);

    BAMutil.addActionToContainer(toolPanel, redrawAction);

    // vertical split
    vertPanel = new VertPanel();
    splitDraw = new JSplitPane(JSplitPane.VERTICAL_SPLIT, panz, vertPanel);
    int divLoc = store.getInt("vertSplit", 2 * defaultHeight / 3);
    splitDraw.setDividerLocation(divLoc);
    drawingPanel = new JPanel(new BorderLayout()); // filled later

    // status panel
    JPanel statusPanel = new JPanel(new BorderLayout());
    statusPanel.setBorder(new EtchedBorder());
    positionLabel = new JLabel("position");
    positionLabel.setToolTipText("position at cursor");
    dataValueLabel = new JLabel("data value", SwingConstants.CENTER);
    dataValueLabel.setToolTipText("data value (double click on grid)");
    statusPanel.add(positionLabel, BorderLayout.WEST);
    statusPanel.add(dataValueLabel, BorderLayout.CENTER);
    panz.setPositionLabel(positionLabel);

    // colorscale panel
    colorScalePanel = new ColorScale.Panel(this, controller.getColorScale());
    csDataMinMax = new JComboBox(ColorScale.MinMaxType.values());
    csDataMinMax.setToolTipText("ColorScale Min/Max setting");
    csDataMinMax
        .addActionListener(e -> controller.setDataMinMaxType((ColorScale.MinMaxType) csDataMinMax.getSelectedItem()));
    JPanel westPanel = new JPanel(new BorderLayout());
    westPanel.add(colorScalePanel, BorderLayout.CENTER);
    westPanel.add(csDataMinMax, BorderLayout.NORTH);

    // lay it out
    JPanel northPanel = new JPanel();
    // northPanel.setLayout( new BoxLayout(northPanel, BoxLayout.Y_AXIS));
    northPanel.setLayout(new BorderLayout());
    northPanel.add(datasetNameLabel, BorderLayout.NORTH);
    northPanel.add(toolPanel, BorderLayout.SOUTH);

    setLayout(new BorderLayout());
    add(northPanel, BorderLayout.NORTH);
    add(statusPanel, BorderLayout.SOUTH);
    add(westPanel, BorderLayout.WEST);
    add(drawingPanel, BorderLayout.CENTER);

    setDrawHorizAndVert(controller.drawHorizOn, controller.drawVertOn);
  }

  private ArrayList<Chooser> choosers;

  private void setChoosers() {
    fieldPanel.removeAll();
    for (Object chooser : choosers) {
      Chooser c = (Chooser) chooser;
      if (c.isWanted) {
        fieldPanel.add(c.field);
      }
    }
  }

  private static class Chooser {
    Chooser(String name, SuperComboBox field, boolean want) {
      this.name = name;
      this.field = field;
      this.isWanted = want;
    }

    boolean isWanted;
    String name;
    SuperComboBox field;
  }

  private void setChooserWanted(String name, boolean want) {
    for (Object chooser1 : choosers) {
      Chooser chooser = (Chooser) chooser1;
      if (chooser.name.equals(name)) {
        chooser.isWanted = want;
      }
    }
  }

  private void addToolbarOption(String toolbarName, JToolBar toolbar, AbstractAction act) {
    boolean wantsToolbar = store.getBoolean(toolbarName, true);
    if (wantsToolbar)
      toolPanel.add(toolbar);
  }

  void addActionsToMenus(JMenu datasetMenu, JMenu configMenu, JMenu toolMenu) {
    // Info
    BAMutil.addActionToMenu(datasetMenu, showGridTableAction);
    BAMutil.addActionToMenu(datasetMenu, showDatasetInfoAction);
    BAMutil.addActionToMenu(datasetMenu, showNcMLAction);
    BAMutil.addActionToMenu(datasetMenu, showGridDatasetInfoAction);
    BAMutil.addActionToMenu(datasetMenu, showNetcdfDatasetAction);
    // BAMutil.addActionToMenu( datasetMenu, geotiffAction);
    // BAMutil.addActionToMenu( infoMenu, showNetcdfXMLAction);

    /// Configure
    JMenu toolbarMenu = new JMenu("Toolbars");
    toolbarMenu.setMnemonic('T');
    configMenu.add(toolbarMenu);
    BAMutil.addActionToMenu(toolbarMenu, navToolbarAction);
    BAMutil.addActionToMenu(toolbarMenu, moveToolbarAction);

    BAMutil.addActionToMenu(configMenu, chooseProjectionAction);
    BAMutil.addActionToMenu(configMenu, saveCurrentProjectionAction);

    /*
     * BAMutil.addActionToMenu( configMenu, chooseColorScaleAction);
     * BAMutil.addActionToMenu( configMenu, controller.dataProjectionAction);
     */

    //// tools menu
    JMenu displayMenu = new JMenu("Display control");
    displayMenu.setMnemonic('D');

    BAMutil.addActionToMenu(displayMenu, controller.showGridAction);
    BAMutil.addActionToMenu(displayMenu, controller.showContoursAction);
    BAMutil.addActionToMenu(displayMenu, controller.showContourLabelsAction);
    BAMutil.addActionToMenu(displayMenu, redrawAction);
    toolMenu.add(displayMenu);

    // Loop Control
    JMenu loopMenu = new JMenu("Loop control");
    loopMenu.setMnemonic('L');

    BAMutil.addActionToMenu(loopMenu, fieldLoopAction);
    BAMutil.addActionToMenu(loopMenu, levelLoopAction);
    BAMutil.addActionToMenu(loopMenu, timeLoopAction);
    toolMenu.add(loopMenu);

    // MinMax Control
    JMenu mmMenu = new JMenu("ColorScale min/max");
    mmMenu.setMnemonic('C');
    BAMutil.addActionToMenu(mmMenu, minmaxHorizAction);
    BAMutil.addActionToMenu(mmMenu, minmaxLogAction);
    BAMutil.addActionToMenu(mmMenu, minmaxHoldAction);
    toolMenu.add(mmMenu);

    // Zoom/Pan
    JMenu zoomMenu = new JMenu("Zoom/Pan");
    zoomMenu.setMnemonic('Z');
    panz.addActionsToMenu(zoomMenu); // items are added by NavigatedPanelToolbar
    toolMenu.add(zoomMenu);
  }

  private static class LoopControlAction extends AbstractAction {
    SuperComboBox scbox;

    LoopControlAction(SuperComboBox cbox) {
      this.scbox = cbox;
      BAMutil.setActionProperties(this, null, cbox.getName(), false, 0, 0);
    }

    public void actionPerformed(ActionEvent e) {
      scbox.getLoopControl().show();
    }
  }

  private class OpenDatasetTask extends ProgressMonitorTask implements ucar.nc2.util.CancelTask {
    DataFactory factory;
    Dataset invds;

    OpenDatasetTask(Dataset ds) {
      factory = new DataFactory();
      this.invds = ds;
    }

    public void run() {
      NetcdfDataset dataset;
      GridDataset gridDataset = null;
      Formatter errlog = new Formatter();

      try {
        dataset = factory.openDataset(invds, true, this, errlog);
        gridDataset = new ucar.nc2.dt.grid.GridDataset(dataset);

      } catch (IOException e) {
        setError("Failed to open datset: " + errlog);
      }

      success = !cancel && (gridDataset != null);
      if (success)
        controller.setGridDataset(gridDataset);
      done = true;
    }
  }

}
