/*
 * Decompiled with CFR 0.152.
 */
package ucar.nc2.util;

import java.io.File;
import java.io.IOException;
import java.io.PrintStream;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.Calendar;
import java.util.Date;
import java.util.Formatter;
import java.util.Random;
import java.util.Timer;
import java.util.TimerTask;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import ucar.nc2.time.CalendarDate;
import ucar.unidata.util.StringUtil2;

public class DiskCache2 {
    private static final Logger cacheLog = LoggerFactory.getLogger("cacheLogger");
    private static final String lockExtension = ".reserve";
    private static Timer timer;
    private CachePathPolicy cachePathPolicy = CachePathPolicy.NestedDirectory;
    private boolean alwaysUseCache;
    private boolean neverUseCache;
    private String cachePathPolicyParam;
    private String root;
    private int persistMinutes;
    private int scourEveryMinutes;
    private boolean fail;

    public static void exit() {
        if (timer != null) {
            timer.cancel();
            cacheLog.info("DiskCache2.exit()%n");
        }
        timer = null;
    }

    private static synchronized void startTimer() {
        if (timer == null) {
            timer = new Timer("DiskCache2");
        }
    }

    public static DiskCache2 getDefault() {
        String root = System.getProperty("nj22.cache");
        if (root == null) {
            String home = System.getProperty("user.home");
            if (home == null) {
                home = System.getProperty("user.dir");
            }
            if (home == null) {
                home = ".";
            }
            root = home + "/.unidata/cache/";
        }
        DiskCache2 result = new DiskCache2();
        result.setRootDirectory(root);
        result.alwaysUseCache = false;
        return result;
    }

    public static DiskCache2 getNoop() {
        DiskCache2 noop = new DiskCache2();
        noop.neverUseCache = true;
        return noop;
    }

    private DiskCache2() {
    }

    public DiskCache2(String root, boolean relativeToHome, int persistMinutes, int scourEveryMinutes) {
        this.persistMinutes = persistMinutes;
        this.scourEveryMinutes = scourEveryMinutes;
        if (relativeToHome) {
            String home = System.getProperty("nj22.cachePersistRoot");
            if (home == null) {
                home = System.getProperty("user.home");
            }
            if (home == null) {
                home = System.getProperty("user.dir");
            }
            if (home == null) {
                home = ".";
            }
            if (!home.endsWith("/")) {
                home = home + "/";
            }
            root = home + root;
        }
        this.setRootDirectory(root);
        if (!this.fail && scourEveryMinutes > 0) {
            DiskCache2.startTimer();
            Calendar c = Calendar.getInstance();
            c.add(12, scourEveryMinutes);
            timer.scheduleAtFixedRate((TimerTask)new CacheScourTask(), c.getTime(), 60000L * (long)scourEveryMinutes);
            cacheLog.info("Started a DiskCache2 scour task on {} every {} minutes for files older than {} minutes", root, scourEveryMinutes, persistMinutes);
        }
    }

    public void setRootDirectory(String cacheDir) {
        if (!cacheDir.endsWith("/")) {
            cacheDir = cacheDir + "/";
        }
        this.root = StringUtil2.replace(cacheDir, '\\', "/");
        File dir = new File(this.root);
        if (!dir.mkdirs()) {
            if (!dir.exists()) {
                cacheLog.warn("Failed to create DiskCache2 root at {}. Reason unknown.", (Object)dir);
            }
        } else {
            cacheLog.debug("DiskCache2 root created at {}.", (Object)dir);
        }
        if (!dir.exists()) {
            this.fail = true;
            cacheLog.error("DiskCache2 failed to create directory " + this.root);
        } else {
            cacheLog.debug("DiskCache2 create directory " + this.root);
        }
    }

    public String getRootDirectory() {
        return this.root;
    }

    public File getCacheFile(String fileLocation) {
        boolean ret;
        File dir;
        File f;
        if (this.neverUseCache) {
            return null;
        }
        if (!this.alwaysUseCache && DiskCache2.canWrite(f = new File(fileLocation))) {
            return f;
        }
        f = new File(this.makeCachePath(fileLocation));
        if (this.cachePathPolicy == CachePathPolicy.NestedDirectory && !(dir = f.getParentFile()).exists() && !(ret = dir.mkdirs())) {
            cacheLog.warn("Error creating dir: " + dir);
        }
        return f;
    }

    public File getFile(String fileLocation) {
        File dir;
        File f;
        if (!this.alwaysUseCache) {
            f = new File(fileLocation);
            if (f.exists()) {
                return f;
            }
            if (DiskCache2.canWrite(f)) {
                return f;
            }
        }
        if (this.neverUseCache) {
            throw new IllegalStateException("neverUseCache=true, but file does not exist and directory is not writeable =" + fileLocation);
        }
        f = new File(this.makeCachePath(fileLocation));
        if (this.cachePathPolicy == CachePathPolicy.NestedDirectory && !(dir = f.getParentFile()).exists() && !dir.mkdirs()) {
            cacheLog.warn("Cant create directories for file " + dir.getPath());
        }
        return f;
    }

    public static boolean canWrite(File f) {
        Path path = f.toPath().toAbsolutePath();
        try {
            if (Files.isDirectory(path, new LinkOption[0])) {
                Files.delete(Files.createTempFile(path, "check", null, new FileAttribute[0]));
            } else if (Files.isRegularFile(path, new LinkOption[0])) {
                Files.newOutputStream(path, StandardOpenOption.APPEND).close();
            } else {
                Files.delete(Files.createTempFile(path.getParent(), "check", null, new FileAttribute[0]));
            }
        }
        catch (IOException | SecurityException e) {
            return false;
        }
        return true;
    }

    public File getExistingFileOrCache(String fileLocation) {
        File f = new File(fileLocation);
        if (f.exists()) {
            return f;
        }
        if (this.neverUseCache) {
            return null;
        }
        File fc = new File(this.makeCachePath(fileLocation));
        if (fc.exists()) {
            return fc;
        }
        return null;
    }

    public synchronized File createUniqueFile(String prefix, String suffix) {
        if (suffix == null) {
            suffix = ".tmp";
        }
        Random random = new Random(System.currentTimeMillis());
        String lockName = prefix + random.nextInt() + suffix + lockExtension;
        File lock = new File(this.getRootDirectory(), lockName);
        while (lock.exists()) {
            lockName = prefix + random.nextInt() + suffix + lockExtension;
            lock = new File(this.getRootDirectory(), lockName);
        }
        try {
            lock.createNewFile();
            cacheLog.debug(String.format("Reserved filename %s for future use.", lockName.replace(lockExtension, "")));
        }
        catch (IOException e) {
            cacheLog.error(String.format("Error creating lock file: %s. May result in cache file collisions.", lock));
        }
        return new File(this.getRootDirectory(), lock.getName().replace(lockExtension, ""));
    }

    public void setPolicy(CachePathPolicy cachePathPolicy) {
        this.cachePathPolicy = cachePathPolicy;
    }

    public void setCachePathPolicy(CachePathPolicy cachePathPolicy, String cachePathPolicyParam) {
        this.cachePathPolicy = cachePathPolicy;
        this.cachePathPolicyParam = cachePathPolicyParam;
    }

    public void setPolicy(String policy) {
        if (policy == null) {
            return;
        }
        if (policy.equalsIgnoreCase("oneDirectory")) {
            this.setCachePathPolicy(CachePathPolicy.OneDirectory, null);
        } else if (policy.equalsIgnoreCase("nestedDirectory")) {
            this.setCachePathPolicy(CachePathPolicy.NestedDirectory, null);
        }
    }

    public void setAlwaysUseCache(boolean alwaysUseCache) {
        this.alwaysUseCache = alwaysUseCache;
    }

    public void setNeverUseCache(boolean neverUseCache) {
        this.neverUseCache = neverUseCache;
    }

    private String makeCachePath(String fileLocation) {
        File file;
        File parent;
        String cachePath = fileLocation;
        cachePath = StringUtil2.remove(cachePath, 63);
        cachePath = StringUtil2.remove(cachePath, 61);
        if ((cachePath = StringUtil2.replace(cachePath, '\\', "/")).startsWith("/")) {
            cachePath = cachePath.substring(1);
        }
        if (cachePath.endsWith("/")) {
            cachePath = cachePath.substring(0, cachePath.length() - 1);
        }
        cachePath = StringUtil2.remove(cachePath, 58);
        if (this.cachePathPolicy == CachePathPolicy.OneDirectory) {
            cachePath = StringUtil2.replace(cachePath, '/', "-");
        } else if (this.cachePathPolicy == CachePathPolicy.NestedTruncate) {
            int pos = cachePath.indexOf(this.cachePathPolicyParam);
            if (pos >= 0) {
                cachePath = cachePath.substring(pos + this.cachePathPolicyParam.length());
            }
            if (cachePath.startsWith("/")) {
                cachePath = cachePath.substring(1);
            }
        }
        if (this.cachePathPolicy != CachePathPolicy.OneDirectory && !(parent = (file = new File(this.root + cachePath)).getParentFile()).exists()) {
            boolean ret;
            if (this.root == null) {
                cacheLog.error("makeCachePath has null root %{}", (Object)parent.getPath());
                new Throwable().printStackTrace();
            }
            if (!(ret = parent.mkdirs())) {
                cacheLog.warn("Error creating parent: " + parent);
            }
        }
        return this.root + cachePath;
    }

    public void showCache(PrintStream pw) {
        pw.println("Cache files");
        pw.println("Size   LastModified       Filename");
        File dir = new File(this.root);
        File[] files = dir.listFiles();
        if (files != null) {
            for (File file : files) {
                String org = null;
                try {
                    org = URLDecoder.decode(file.getName(), "UTF8");
                }
                catch (UnsupportedEncodingException e) {
                    e.printStackTrace();
                }
                pw.println(" " + file.length() + " " + new Date(file.lastModified()) + " " + org);
            }
        }
    }

    public void cleanCache(File dir, Formatter sbuff, boolean isRoot) {
        long now = System.currentTimeMillis();
        File[] files = dir.listFiles();
        if (files == null) {
            throw new IllegalStateException("DiskCache2: not a directory or I/O error on dir=" + dir.getAbsolutePath());
        }
        if (!isRoot && files.length == 0) {
            this.purgeExpired(dir, sbuff, now);
            return;
        }
        for (File file : files) {
            if (file.isDirectory()) {
                this.cleanCache(file, sbuff, false);
                continue;
            }
            this.purgeExpired(file, sbuff, now);
        }
    }

    private void purgeExpired(File deletable, Formatter sbuff, long now) {
        long duration = now - deletable.lastModified();
        if ((duration /= 60000L) > (long)this.persistMinutes) {
            boolean ok = deletable.delete();
            if (!ok) {
                String type = deletable.isDirectory() ? "directory" : "file";
                cacheLog.error("Unable to delete {} {}", (Object)type, (Object)deletable.getAbsolutePath());
            }
            if (sbuff != null) {
                sbuff.format(" deleted %s %s lastModified= %s%n", ok, deletable.getPath(), CalendarDate.of(deletable.lastModified()));
            }
        }
    }

    public String toString() {
        String sb = "DiskCache2{cachePathPolicy=" + (Object)((Object)this.cachePathPolicy) + ", alwaysUseCache=" + this.alwaysUseCache + ", cachePathPolicyParam='" + this.cachePathPolicyParam + '\'' + ", root='" + this.root + '\'' + ", scourEveryMinutes=" + this.scourEveryMinutes + ", persistMinutes=" + this.persistMinutes + ", fail=" + this.fail + '}';
        return sb;
    }

    public static enum CachePathPolicy {
        OneDirectory,
        NestedDirectory,
        NestedTruncate;

    }

    private class CacheScourTask
    extends TimerTask {
        private CacheScourTask() {
        }

        @Override
        public void run() {
            Formatter sbuff = new Formatter();
            sbuff.format("DiskCache2 scour on directory= %s%n", DiskCache2.this.root);
            DiskCache2.this.cleanCache(new File(DiskCache2.this.root), sbuff, true);
            if (cacheLog.isDebugEnabled()) {
                cacheLog.debug(sbuff.toString());
            }
        }
    }
}

