/*
 * Decompiled with CFR 0.152.
 */
package ucar.nc2.util.cache;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.Formatter;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import net.jcip.annotations.GuardedBy;
import net.jcip.annotations.ThreadSafe;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import ucar.nc2.util.CancelTask;
import ucar.nc2.util.cache.FileCacheable;
import ucar.nc2.util.cache.FileFactory;

@ThreadSafe
public class FileCache {
    private static final Logger log = LoggerFactory.getLogger(FileCache.class);
    private static final Logger cacheLog = LoggerFactory.getLogger((String)"cacheLogger");
    private static ScheduledExecutorService exec;
    static boolean debug;
    static boolean debugPrint;
    static boolean debugCleanup;
    private String name;
    private final int softLimit;
    private final int minElements;
    private final int hardLimit;
    private final ConcurrentHashMap<Object, CacheElement> cache;
    private final ConcurrentHashMap<FileCacheable, CacheElement.CacheFile> files;
    private final AtomicBoolean hasScheduled = new AtomicBoolean(false);
    private final AtomicBoolean disabled = new AtomicBoolean(false);
    private final AtomicInteger cleanups = new AtomicInteger();
    private final AtomicInteger hits = new AtomicInteger();
    private final AtomicInteger miss = new AtomicInteger();

    public static void shutdown() {
        if (exec != null) {
            exec.shutdown();
        }
        exec = null;
    }

    public FileCache(int minElementsInMemory, int maxElementsInMemory, int period) {
        this("", minElementsInMemory, maxElementsInMemory, -1, period);
    }

    public FileCache(int minElementsInMemory, int softLimit, int hardLimit, int period) {
        this("", minElementsInMemory, softLimit, hardLimit, period);
    }

    public FileCache(String name, int minElementsInMemory, int softLimit, int hardLimit, int period) {
        this.name = name;
        this.minElements = minElementsInMemory;
        this.softLimit = softLimit;
        this.hardLimit = hardLimit;
        this.cache = new ConcurrentHashMap(2 * softLimit, 0.75f, 8);
        this.files = new ConcurrentHashMap(4 * softLimit, 0.75f, 8);
        if (period > 0) {
            if (exec == null) {
                exec = Executors.newSingleThreadScheduledExecutor();
            }
            exec.scheduleAtFixedRate(new CleanupTask(), period, period, TimeUnit.SECONDS);
            cacheLog.debug("FileCache " + name + " cleanup every " + period + " secs");
        }
    }

    public void disable() {
        this.disabled.set(true);
        this.clearCache(true);
    }

    public void enable() {
        this.disabled.set(false);
    }

    public FileCacheable acquire(FileFactory factory, String location, CancelTask cancelTask) throws IOException {
        return this.acquire(factory, location, location, -1, cancelTask, null);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public FileCacheable acquire(FileFactory factory, Object hashKey, String location, int buffer_size, CancelTask cancelTask, Object spiObject) throws IOException {
        CacheElement elem;
        FileCacheable ncfile;
        if (null == hashKey) {
            hashKey = location;
        }
        if ((ncfile = this.acquireCacheOnly(hashKey)) != null) {
            this.hits.incrementAndGet();
            return ncfile;
        }
        this.miss.incrementAndGet();
        ncfile = factory.open(location, buffer_size, cancelTask, spiObject);
        if (cacheLog.isDebugEnabled()) {
            cacheLog.debug("FileCache " + this.name + " acquire " + hashKey + " " + ncfile.getLocation());
        }
        if (debugPrint) {
            System.out.println("  FileCache " + this.name + " acquire " + hashKey + " " + ncfile.getLocation());
        }
        if (cancelTask != null && cancelTask.isCancel()) {
            if (ncfile != null) {
                ncfile.close();
            }
            return null;
        }
        if (this.disabled.get()) {
            return ncfile;
        }
        Object object = this.cache;
        synchronized (object) {
            elem = this.cache.get(hashKey);
            if (elem == null) {
                this.cache.put(hashKey, new CacheElement(ncfile, hashKey));
            }
        }
        if (elem != null) {
            object = elem;
            synchronized (object) {
                elem.addFile(ncfile);
            }
        }
        boolean needHard = false;
        boolean needSoft = false;
        AtomicBoolean atomicBoolean = this.hasScheduled;
        synchronized (atomicBoolean) {
            if (!this.hasScheduled.get()) {
                int count = this.files.size();
                if (count > this.hardLimit && this.hardLimit > 0) {
                    needHard = true;
                    this.hasScheduled.getAndSet(true);
                } else if (count > this.softLimit && exec != null) {
                    this.hasScheduled.getAndSet(true);
                    needSoft = true;
                }
            }
        }
        if (needHard) {
            if (debugCleanup) {
                System.out.println("CleanupTask due to hard limit time=" + new Date().getTime());
            }
            this.cleanup(this.hardLimit);
        } else if (needSoft) {
            exec.schedule(new CleanupTask(), 100L, TimeUnit.MILLISECONDS);
            if (debugCleanup) {
                System.out.println("CleanupTask scheduled due to soft limit time=" + new Date());
            }
        }
        return ncfile;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private FileCacheable acquireCacheOnly(Object hashKey) {
        if (this.disabled.get()) {
            return null;
        }
        FileCacheable ncfile = null;
        CacheElement elem = this.cache.get(hashKey);
        if (elem != null) {
            CacheElement cacheElement = elem;
            synchronized (cacheElement) {
                for (CacheElement.CacheFile file : elem.list) {
                    if (!file.isLocked.compareAndSet(false, true)) continue;
                    ncfile = file.ncfile;
                    break;
                }
            }
        }
        if (ncfile != null) {
            try {
                boolean changed = ncfile.sync();
                if (cacheLog.isDebugEnabled()) {
                    cacheLog.debug("FileCache " + this.name + " aquire from cache " + hashKey + " " + ncfile.getLocation() + " changed = " + changed);
                }
                if (debugPrint) {
                    System.out.println("  FileCache " + this.name + " aquire from cache " + hashKey + " " + ncfile.getLocation() + " changed = " + changed);
                }
            }
            catch (IOException e) {
                log.error("FileCache " + this.name + " synch failed on " + ncfile.getLocation() + " " + e.getMessage());
                return null;
            }
        }
        return ncfile;
    }

    public void release(FileCacheable ncfile) throws IOException {
        if (ncfile == null) {
            return;
        }
        if (this.disabled.get()) {
            ncfile.setFileCache(null);
            ncfile.close();
            return;
        }
        CacheElement.CacheFile file = this.files.get(ncfile);
        if (file != null) {
            if (!file.isLocked.get()) {
                cacheLog.warn("FileCache " + this.name + " release " + ncfile.getLocation() + " not locked");
            }
            file.lastAccessed = System.currentTimeMillis();
            ++file.countAccessed;
            file.isLocked.set(false);
            if (cacheLog.isDebugEnabled()) {
                cacheLog.debug("FileCache " + this.name + " release " + ncfile.getLocation());
            }
            if (debugPrint) {
                System.out.println("  FileCache " + this.name + " release " + ncfile.getLocation());
            }
            return;
        }
        throw new IOException("FileCache " + this.name + " release does not have file in cache = " + ncfile.getLocation());
    }

    public String getInfo(FileCacheable ncfile) throws IOException {
        if (ncfile == null) {
            return "";
        }
        CacheElement.CacheFile file = this.files.get(ncfile);
        if (file != null) {
            return "File is in cache= " + file;
        }
        return "File not in cache";
    }

    Map<Object, CacheElement> getCache() {
        return this.cache;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public synchronized void clearCache(boolean force) {
        ArrayList<Object> deleteList = new ArrayList<Object>(2 * this.cache.size());
        if (force) {
            this.cache.clear();
            deleteList.addAll(this.files.values());
            this.files.clear();
        } else {
            Object file;
            Iterator<CacheElement.CacheFile> iter = this.files.values().iterator();
            while (iter.hasNext()) {
                file = iter.next();
                if (!((CacheElement.CacheFile)file).isLocked.compareAndSet(false, true)) continue;
                ((CacheElement.CacheFile)file).remove();
                deleteList.add(file);
                iter.remove();
            }
            file = this.cache;
            synchronized (file) {
                Iterator<CacheElement> i$ = this.cache.values().iterator();
                while (i$.hasNext()) {
                    CacheElement elem;
                    CacheElement cacheElement = elem = i$.next();
                    synchronized (cacheElement) {
                        if (elem.list.size() == 0) {
                            this.cache.remove(elem.hashKey);
                        }
                    }
                }
            }
        }
        for (Object file : deleteList) {
            if (force && ((CacheElement.CacheFile)file).isLocked.get()) {
                cacheLog.warn("FileCache " + this.name + " force close locked file= " + file);
            }
            try {
                ((CacheElement.CacheFile)file).ncfile.setFileCache(null);
                ((CacheElement.CacheFile)file).ncfile.close();
                ((CacheElement.CacheFile)file).ncfile = null;
            }
            catch (IOException e) {
                log.error("FileCache " + this.name + " close failed on " + file);
            }
        }
        cacheLog.debug("*FileCache " + this.name + " clearCache force= " + force + " deleted= " + deleteList.size() + " left=" + this.files.size());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void showCache(Formatter format) {
        ArrayList<CacheElement.CacheFile> allFiles = new ArrayList<CacheElement.CacheFile>(this.files.size());
        Iterator<CacheElement> i$ = this.cache.values().iterator();
        while (i$.hasNext()) {
            CacheElement elem;
            CacheElement cacheElement = elem = i$.next();
            synchronized (cacheElement) {
                allFiles.addAll(elem.list);
            }
        }
        Collections.sort(allFiles);
        format.format("FileCache %s (%d):%n", this.name, allFiles.size());
        format.format("isLocked  accesses lastAccess                   location %n", new Object[0]);
        for (CacheElement.CacheFile file : allFiles) {
            String loc = file.ncfile != null ? file.ncfile.getLocation() : "null";
            format.format("%8s %9d %s %s %n", file.isLocked, file.countAccessed, new Date(file.lastAccessed), loc);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<String> showCache() {
        ArrayList<CacheElement.CacheFile> allFiles = new ArrayList<CacheElement.CacheFile>(this.files.size());
        Iterator<CacheElement> i$ = this.cache.values().iterator();
        while (i$.hasNext()) {
            CacheElement elem;
            CacheElement cacheElement = elem = i$.next();
            synchronized (cacheElement) {
                allFiles.addAll(elem.list);
            }
        }
        Collections.sort(allFiles);
        ArrayList<String> result = new ArrayList<String>(allFiles.size());
        for (CacheElement.CacheFile file : allFiles) {
            result.add(file.toString());
        }
        return result;
    }

    public void showStats(Formatter format) {
        format.format("  hits= %d miss= %d nfiles= %d elems= %d\n", this.hits.get(), this.miss.get(), this.files.size(), this.cache.values().size());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    synchronized void cleanup(int maxElements) {
        if (this.disabled.get()) {
            return;
        }
        try {
            Object file;
            int size = this.files.size();
            if (size <= this.minElements) {
                return;
            }
            cacheLog.debug(" FileCache " + this.name + " cleanup started at " + new Date() + " for cleanup maxElements=" + maxElements);
            if (debugCleanup) {
                System.out.println(" FileCache " + this.name + "cleanup started at " + new Date() + " for cleanup maxElements=" + maxElements);
            }
            this.cleanups.incrementAndGet();
            ArrayList<CacheElement.CacheFile> allFiles = new ArrayList<CacheElement.CacheFile>(size + 10);
            for (CacheElement.CacheFile file2 : this.files.values()) {
                if (file2.isLocked.get()) continue;
                allFiles.add(file2);
            }
            Collections.sort(allFiles);
            int need2delete = size - this.minElements;
            int minDelete = size - maxElements;
            ArrayList<Object> deleteList = new ArrayList<Object>(need2delete);
            int count = 0;
            Iterator iter = allFiles.iterator();
            while (iter.hasNext() && count < need2delete) {
                file = (CacheElement.CacheFile)iter.next();
                if (!((CacheElement.CacheFile)file).isLocked.compareAndSet(false, true)) continue;
                ((CacheElement.CacheFile)file).remove();
                deleteList.add(file);
                ++count;
            }
            if (count < minDelete) {
                cacheLog.warn("FileCache " + this.name + " cleanup couldnt remove enough to keep under the maximum= " + maxElements + " due to locked files; currently at = " + (size - count));
                if (debugCleanup) {
                    System.out.println("FileCache " + this.name + "cleanup couldnt remove enough to keep under the maximum= " + maxElements + " due to locked files; currently at = " + (size - count));
                }
            }
            file = this.cache;
            synchronized (file) {
                Iterator<CacheElement> i$ = this.cache.values().iterator();
                while (i$.hasNext()) {
                    CacheElement elem;
                    CacheElement cacheElement = elem = i$.next();
                    synchronized (cacheElement) {
                        if (elem.list.size() == 0) {
                            this.cache.remove(elem.hashKey);
                        }
                    }
                }
            }
            long start = System.currentTimeMillis();
            for (CacheElement.CacheFile cacheFile : deleteList) {
                this.files.remove(cacheFile.ncfile);
                try {
                    cacheFile.ncfile.setFileCache(null);
                    cacheFile.ncfile.close();
                    cacheFile.ncfile = null;
                }
                catch (IOException e) {
                    log.error("FileCache " + this.name + " close failed on " + cacheFile.getCacheName());
                }
            }
            long took = System.currentTimeMillis() - start;
            cacheLog.debug(" FileCache " + this.name + " cleanup had= " + size + " removed= " + deleteList.size() + " took=" + took + " msec");
            if (debugCleanup) {
                System.out.println(" FileCache " + this.name + "cleanup had= " + size + " removed= " + deleteList.size() + " took=" + took + " msec");
            }
        }
        finally {
            this.hasScheduled.set(false);
        }
    }

    static {
        debug = false;
        debugPrint = false;
        debugCleanup = false;
    }

    private class CleanupTask
    implements Runnable {
        private CleanupTask() {
        }

        @Override
        public void run() {
            FileCache.this.cleanup(FileCache.this.softLimit);
        }
    }

    class CacheElement {
        @GuardedBy(value="this")
        List<CacheFile> list = new LinkedList<CacheFile>();
        final Object hashKey;

        CacheElement(FileCacheable ncfile, Object hashKey) {
            this.hashKey = hashKey;
            CacheFile file = new CacheFile(ncfile);
            this.list.add(file);
            if (debug && FileCache.this.files.get(ncfile) != null) {
                cacheLog.error("files already has " + hashKey + " " + FileCache.this.name);
            }
            FileCache.this.files.put(ncfile, file);
            if (cacheLog.isDebugEnabled()) {
                cacheLog.debug("CacheElement add to cache " + hashKey + " " + FileCache.this.name);
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        CacheFile addFile(FileCacheable ncfile) {
            CacheFile file = new CacheFile(ncfile);
            CacheElement cacheElement = this;
            synchronized (cacheElement) {
                this.list.add(file);
            }
            if (debug && FileCache.this.files.get(ncfile) != null) {
                cacheLog.error("files (2) already has " + this.hashKey + " " + FileCache.this.name);
            }
            FileCache.this.files.put(ncfile, file);
            return file;
        }

        public String toString() {
            return this.hashKey + " count=" + this.list.size();
        }

        class CacheFile
        implements Comparable<CacheFile> {
            FileCacheable ncfile;
            final AtomicBoolean isLocked = new AtomicBoolean(true);
            int countAccessed = 1;
            long lastAccessed = 0L;

            private CacheFile(FileCacheable ncfile) {
                this.ncfile = ncfile;
                this.lastAccessed = System.currentTimeMillis();
                ncfile.setFileCache(FileCache.this);
                if (cacheLog.isDebugEnabled()) {
                    cacheLog.debug("FileCache " + FileCache.this.name + " add to cache " + CacheElement.this.hashKey);
                }
                if (debugPrint) {
                    System.out.println("  FileCache " + FileCache.this.name + " add to cache " + CacheElement.this.hashKey);
                }
            }

            String getCacheName() {
                return this.ncfile.getLocation();
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            void remove() {
                CacheElement cacheElement = CacheElement.this;
                synchronized (cacheElement) {
                    if (!CacheElement.this.list.remove(this)) {
                        cacheLog.warn("FileCache " + FileCache.this.name + " could not remove " + this.ncfile.getLocation());
                    }
                }
                if (cacheLog.isDebugEnabled()) {
                    cacheLog.debug("FileCache " + FileCache.this.name + " remove " + this.ncfile.getLocation());
                }
                if (debugPrint) {
                    System.out.println("  FileCache " + FileCache.this.name + " remove " + this.ncfile.getLocation());
                }
            }

            public String toString() {
                return this.isLocked + " " + this.countAccessed + " " + new Date(this.lastAccessed) + " " + this.ncfile.getLocation();
            }

            @Override
            public int compareTo(CacheFile o) {
                return (int)(this.lastAccessed - o.lastAccessed);
            }
        }
    }
}

