/*
 * Decompiled with CFR 0.152.
 */
package thredds.cache;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.time.Duration;
import java.util.Formatter;
import java.util.Map;
import java.util.concurrent.atomic.AtomicLong;
import org.eclipse.serializer.collections.lazy.LazyHashMap;
import org.eclipse.serializer.concurrency.XThreads;
import org.eclipse.serializer.persistence.binary.jdk17.types.BinaryHandlersJDK17;
import org.eclipse.serializer.persistence.binary.jdk8.types.BinaryHandlersJDK8;
import org.eclipse.store.storage.embedded.configuration.types.EmbeddedStorageConfiguration;
import org.eclipse.store.storage.embedded.types.EmbeddedStorageFoundation;
import org.eclipse.store.storage.embedded.types.EmbeddedStorageManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class DiskPersistedCache<K, V> {
    private static final Logger logger = LoggerFactory.getLogger(DiskPersistedCache.class);
    private final EmbeddedStorageManager storageManager;
    private final Cache<K, V> level1Cache;
    private final LazyHashMap<K, V> level2Cache;

    private DiskPersistedCache(Builder<K, V> builder) {
        this.storageManager = builder.storageManager;
        this.level2Cache = builder.level2Cache;
        this.level1Cache = builder.level1Cache;
    }

    public static <K, V> Builder<K, V> at(Path cacheDir) throws IOException {
        return new Builder(cacheDir);
    }

    public V get(K key) {
        Object val = this.level1Cache.getIfPresent(key);
        if (val == null) {
            logger.debug("{} L1 miss", (Object)this.storageManager.databaseName());
            val = this.level2Cache.get(key);
            if (val != null) {
                logger.debug("{} L2 cache hit, update L1 cache", (Object)this.storageManager.databaseName());
                this.level1Cache.put(key, val);
            } else {
                logger.debug("{} L2 cache miss", (Object)this.storageManager.databaseName());
            }
        } else {
            logger.debug("{} L1 cache hit", (Object)this.storageManager.databaseName());
        }
        if (val == null) {
            logger.debug("{} key {} not in cache", (Object)this.storageManager.databaseName(), key);
        }
        return (V)val;
    }

    public V getOrDefault(K key, V defaultValue) {
        V val = this.get(key);
        return val == null ? defaultValue : val;
    }

    public void put(K key, V val) {
        XThreads.executeSynchronized(() -> {
            logger.debug("{} adding value to L2 cache", (Object)this.storageManager.databaseName());
            this.level2Cache.put(key, val);
            this.storageManager.store(this.level2Cache);
            logger.debug("{} updating L1 cache", (Object)this.storageManager.databaseName());
            this.level1Cache.put(key, val);
        });
    }

    public void remove(K key) {
        XThreads.executeSynchronized(() -> {
            this.level2Cache.remove(key);
            this.storageManager.store(this.level2Cache);
            this.level1Cache.invalidate(key);
        });
    }

    public long numL1Keys() {
        return this.level1Cache.size();
    }

    public long numL2Keys() {
        return this.level2Cache.size();
    }

    public void reinit() {
        if (this.storageManager.isRunning()) {
            this.level2Cache.clear();
            this.cleanupL2Storage();
        } else {
            logger.warn("{} is not running, cannot reinit l2 cache", (Object)this.storageManager.databaseName());
        }
        this.level1Cache.invalidateAll();
    }

    public String name() {
        return this.storageManager.databaseName();
    }

    public boolean running() {
        return this.storageManager.isRunning();
    }

    public void showL1Db(Formatter f) {
        long count = 0L;
        for (Map.Entry entry : this.level1Cache.asMap().entrySet()) {
            f.format("%4d: '%s' == %s%n", count++, entry.getKey(), entry.getValue());
        }
    }

    public void showL2Db(Formatter f, int skip) {
        if (skip < 1) {
            skip = 1;
        }
        int peekSkip = skip;
        AtomicLong count = new AtomicLong();
        count.set(-1L);
        for (Map.Entry entry : this.level2Cache.entrySet()) {
            long itemNum = count.incrementAndGet();
            if (itemNum % (long)peekSkip != 0L) continue;
            f.format("%4d: '%s' == %s%n", itemNum, entry.getKey(), entry.getValue());
        }
    }

    public void shutdown() {
        logger.info("shutting down {} cache...", (Object)this.storageManager.databaseName());
        this.storageManager.shutdown();
        logger.info("{} cache shutdown", (Object)this.storageManager.databaseName());
    }

    void cleanupL2Storage() {
        this.storageManager.issueFullCacheCheck();
        this.storageManager.issueFullFileCheck();
    }

    V getL1Cache(K key) {
        return (V)this.level1Cache.getIfPresent(key);
    }

    V getL2Cache(K key) {
        return (V)this.level2Cache.get(key);
    }

    public static class Builder<T1, T2> {
        private final int entitiesPerSegment = 10;
        private final LazyHashMap<T1, T2> level2Cache = new LazyHashMap(10);
        private final Path cacheDir;
        private EmbeddedStorageManager storageManager;
        private Cache<T1, T2> level1Cache;
        private long maximumL1CacheEntities = 1000L;
        private String name;

        private Builder(Path cacheDir) throws IOException {
            if (Files.exists(cacheDir, new LinkOption[0])) {
                if (!Files.isDirectory(cacheDir, new LinkOption[0])) {
                    throw new IOException(String.format("cacheDir %s is not a directory", cacheDir));
                }
                if (!Files.isReadable(cacheDir)) {
                    throw new IOException(String.format("cacheDir %s is not readable", cacheDir));
                }
                if (!Files.isWritable(cacheDir)) {
                    throw new IOException(String.format("cacheDir %s is not writable", cacheDir));
                }
            }
            this.cacheDir = cacheDir.toAbsolutePath();
            this.name = this.cacheDir.getFileName().toString();
        }

        public Builder<T1, T2> named(String cacheName) {
            this.name = cacheName;
            return this;
        }

        public Builder<T1, T2> maxInMemoryEntities(long maximumMemoryCacheEntities) {
            this.maximumL1CacheEntities = maximumMemoryCacheEntities;
            return this;
        }

        public DiskPersistedCache<T1, T2> build() throws IOException {
            if (!Files.exists(this.cacheDir, new LinkOption[0])) {
                logger.info("creating cache directory at {}", (Object)this.cacheDir);
                Files.createDirectories(this.cacheDir, new FileAttribute[0]);
            }
            Duration lazyRefTimeout = Duration.ofSeconds(30L);
            EmbeddedStorageFoundation foundation = EmbeddedStorageConfiguration.Builder().setStorageDirectory(this.cacheDir.toAbsolutePath().toString()).setEntityCacheTimeout(lazyRefTimeout).createEmbeddedStorageFoundation().setDataBaseName(this.name).setRoot(this.level2Cache);
            foundation.onConnectionFoundation(BinaryHandlersJDK8::registerJDK8TypeHandlers);
            foundation.onConnectionFoundation(BinaryHandlersJDK17::registerJDK17TypeHandlers);
            this.storageManager = foundation.start();
            logger.info("{} cache stored at {}", (Object)this.storageManager.databaseName(), (Object)this.cacheDir);
            this.level1Cache = CacheBuilder.newBuilder().maximumSize(this.maximumL1CacheEntities).build();
            return new DiskPersistedCache(this);
        }
    }
}

