/*
 * Decompiled with CFR 0.152.
 */
package edu.ucar.unidata.sruth;

import edu.ucar.unidata.sruth.ArchivePath;
import edu.ucar.unidata.sruth.ArchiveTime;
import edu.ucar.unidata.sruth.CompleteBitSet;
import edu.ucar.unidata.sruth.DataProduct;
import edu.ucar.unidata.sruth.DataProductListener;
import edu.ucar.unidata.sruth.DelayedPathActionQueue;
import edu.ucar.unidata.sruth.FileId;
import edu.ucar.unidata.sruth.FileInfo;
import edu.ucar.unidata.sruth.FileInfoMismatchException;
import edu.ucar.unidata.sruth.FilePieceSpecSet;
import edu.ucar.unidata.sruth.FilePieceSpecSetConsumer;
import edu.ucar.unidata.sruth.Filter;
import edu.ucar.unidata.sruth.FiniteBitSet;
import edu.ucar.unidata.sruth.ObjectLock;
import edu.ucar.unidata.sruth.PartialBitSet;
import edu.ucar.unidata.sruth.PathDelayQueue;
import edu.ucar.unidata.sruth.Piece;
import edu.ucar.unidata.sruth.PieceSpec;
import edu.ucar.unidata.sruth.Server;
import edu.ucar.unidata.sruth.Topology;
import edu.ucar.unidata.sruth.Util;
import java.io.ByteArrayInputStream;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOError;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.ObjectStreamException;
import java.io.RandomAccessFile;
import java.io.Serializable;
import java.io.StreamCorruptedException;
import java.net.InetSocketAddress;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.ReadableByteChannel;
import java.nio.file.DirectoryNotEmptyException;
import java.nio.file.DirectoryStream;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.FileSystemException;
import java.nio.file.FileVisitOption;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.locks.ReentrantLock;
import java.util.prefs.Preferences;
import net.jcip.annotations.GuardedBy;
import net.jcip.annotations.NotThreadSafe;
import net.jcip.annotations.ThreadSafe;
import org.slf4j.Logger;

@ThreadSafe
final class Archive {
    private static final Logger logger = Util.getLogger();
    private static final Path HIDDEN_DIR = Paths.get(".sruth", new String[0]);
    private static final int PIECE_SIZE = 131072;
    private static final int ACTIVE_FILE_CACHE_SIZE;
    private static final int ACTIVE_FILE_CACHE_SIZE_DEFAULT = 512;
    private static final String ACTIVE_FILE_CACHE_SIZE_KEY = "active file cache size";
    private final Path rootDir;
    private final DelayedPathActionQueue delayedPathActionQueue;
    @GuardedBy(value="itself")
    private final List<DataProductListener> dataProductListeners = new LinkedList<DataProductListener>();
    private final DistributedTrackerFilesFactory distributedTrackerFilesFactory = new DistributedTrackerFilesFactory();
    private final ArchivePath adminDir = new ArchivePath(Util.PACKAGE_NAME);
    private final ArchiveFileManager archiveFileManager;

    Archive(String rootDir) throws IOException {
        this(Paths.get(rootDir, new String[0]));
    }

    Archive(Path rootDir) throws IOException {
        this(rootDir, ACTIVE_FILE_CACHE_SIZE);
    }

    Archive(final Path rootDir, int maxNumOpenFiles) throws IOException {
        if (null == rootDir) {
            throw new NullPointerException();
        }
        if (maxNumOpenFiles <= 0) {
            throw new IllegalArgumentException("Invalid maximum number of open file: " + maxNumOpenFiles);
        }
        Path hiddenDir = rootDir.resolve(HIDDEN_DIR);
        Path fileDeletionQueuePath = hiddenDir.resolve("fileDeletionQueue");
        Files.createDirectories(hiddenDir, new FileAttribute[0]);
        Archive.purgeHiddenDir(hiddenDir, fileDeletionQueuePath);
        try {
            Boolean hidden = (Boolean)Files.getAttribute(hiddenDir, "dos:hidden", LinkOption.NOFOLLOW_LINKS);
            if (null != hidden && !hidden.booleanValue()) {
                Files.setAttribute(hiddenDir, "dos:hidden", Boolean.TRUE, LinkOption.NOFOLLOW_LINKS);
            }
        }
        catch (FileSystemException ignored) {
            // empty catch block
        }
        this.rootDir = rootDir;
        this.archiveFileManager = new ArchiveFileManager(maxNumOpenFiles);
        this.delayedPathActionQueue = new DelayedPathActionQueue(rootDir, new PathDelayQueue(fileDeletionQueuePath), new DelayedPathActionQueue.Action(){

            @Override
            void act(Path path) throws IOException {
                Archive.this.archiveFileManager.delete(new ArchivePath(path, rootDir));
            }

            public String toString() {
                return "DELETE";
            }
        });
    }

    private static void purgeHiddenDir(final Path hiddenDir, final Path keepPath) throws IOException {
        EnumSet<FileVisitOption> opts = EnumSet.of(FileVisitOption.FOLLOW_LINKS);
        Files.walkFileTree(hiddenDir, opts, Integer.MAX_VALUE, new SimpleVisitor(){

            @Override
            public FileVisitResult visitFile(Path path, BasicFileAttributes attributes) throws IOException {
                if (!path.equals(keepPath)) {
                    try {
                        Files.delete(path);
                    }
                    catch (IOException e) {
                        logger.error("Couldn't purge file: " + path, e);
                        throw e;
                    }
                }
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult postVisitDirectory(Path dir, IOException e) throws IOException {
                if (e != null) {
                    throw e;
                }
                if (!dir.equals(hiddenDir)) {
                    try {
                        Files.delete(dir);
                    }
                    catch (IOException e2) {
                        logger.error("Couldn't purge directory: " + dir, e2);
                        throw e2;
                    }
                }
                return FileVisitResult.CONTINUE;
            }
        });
    }

    Path getRootDir() {
        return this.rootDir;
    }

    ArchivePath getAdminDir() {
        return this.adminDir;
    }

    ArchivePath relativize(Path path) {
        return new ArchivePath(path, this.rootDir);
    }

    Path resolve(ArchivePath path) {
        return path.getAbsolutePath(this.rootDir);
    }

    DistributedTrackerFiles getDistributedTrackerFiles(InetSocketAddress trackerAddress) {
        return this.distributedTrackerFilesFactory.getInstance(trackerAddress);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    boolean exists(PieceSpec pieceSpec) throws FileSystemException, IOException {
        SegmentedArchiveFile file = this.archiveFileManager.get(pieceSpec.getFileInfo(), true);
        if (file == null) {
            return false;
        }
        try {
            boolean bl = file.hasPiece(pieceSpec.getIndex());
            file.unlock();
            return bl;
        }
        catch (Throwable throwable) {
            try {
                file.unlock();
                throw throwable;
            }
            catch (FileInfoMismatchException e) {
                logger.debug("Incompatible file-information: {}", (Object)e.toString());
                return false;
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    Piece getPiece(PieceSpec pieceSpec) throws FileSystemException, IOException {
        SegmentedArchiveFile file;
        try {
            file = this.archiveFileManager.get(pieceSpec.getFileInfo(), true);
        }
        catch (FileInfoMismatchException e) {
            logger.warn("Incompatible file-information", e);
            return null;
        }
        if (file == null) {
            return null;
        }
        try {
            Piece piece = file.getPiece(pieceSpec);
            return piece;
        }
        finally {
            file.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    boolean putPiece(Piece piece) throws FileSystemException, NoSuchFileException, FileInfoMismatchException, IOException {
        FileInfo fileInfo = piece.getFileInfo();
        SegmentedArchiveFile file = this.archiveFileManager.get(fileInfo, false);
        if (file == null) {
            logger.trace("Newer file version exists: {}", fileInfo);
            return false;
        }
        try {
            boolean isComplete;
            int timeToLive = piece.getTimeToLive();
            if (timeToLive >= 0 && file.willMakeComplete(piece.getInfo())) {
                this.delayedPathActionQueue.actUponEventurally(fileInfo.getAbsolutePath(this.rootDir), 1000 * timeToLive);
            }
            if (isComplete = file.putPiece(piece)) {
                List<DataProductListener> list = this.dataProductListeners;
                synchronized (list) {
                    for (DataProductListener listener : this.dataProductListeners) {
                        DataProduct product = new DataProduct(this.rootDir, fileInfo);
                        listener.process(product);
                    }
                }
            }
            boolean bl = isComplete;
            return bl;
        }
        finally {
            file.unlock();
        }
    }

    private void save(ArchivePath archivePath, Serializable serializable) throws FileSystemException, IOException {
        byte[] bytes = Util.serialize(serializable);
        ByteArrayInputStream inputStream = new ByteArrayInputStream(bytes);
        ReadableByteChannel channel = Channels.newChannel(inputStream);
        this.save(archivePath, channel);
    }

    void save(ArchivePath path, ByteBuffer byteBuf) throws FileAlreadyExistsException, IOException, FileInfoMismatchException {
        byte[] bytes = byteBuf.array();
        ByteArrayInputStream inputStream = new ByteArrayInputStream(bytes);
        ReadableByteChannel channel = Channels.newChannel(inputStream);
        this.save(path, channel);
    }

    void save(ArchivePath path, ReadableByteChannel channel) throws FileSystemException, IOException {
        this.save(path, channel, -1);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void save(ArchivePath archivePath, ReadableByteChannel channel, int timeToLive) throws FileSystemException, IOException {
        BulkArchiveFile file = this.archiveFileManager.getForWriting(archivePath);
        boolean success = false;
        try {
            file.transferFrom(channel);
            if (timeToLive >= 0) {
                this.delayedPathActionQueue.actUponEventurally(archivePath.getAbsolutePath(this.rootDir), 1000 * timeToLive);
            }
            success = true;
        }
        finally {
            file.close();
            if (!success) {
                this.archiveFileManager.delete(archivePath);
            }
        }
    }

    Object restore(ArchivePath archivePath, Class<?> type) throws FileNotFoundException, StreamCorruptedException, ClassNotFoundException, IOException {
        BulkArchiveFile file = this.archiveFileManager.getForReading(archivePath);
        boolean success = false;
        try {
            InputStream in = file.asInputStream();
            ObjectInputStream ois = new ObjectInputStream(in);
            try {
                Object obj = ois.readObject();
                if (!type.isInstance(obj)) {
                    throw new ClassCastException("expected=" + type.getSimpleName() + ", actual=" + obj.getClass());
                }
                success = true;
                Object object = obj;
                return object;
            }
            catch (StreamCorruptedException e) {
                throw (StreamCorruptedException)new StreamCorruptedException("Corrupted file: " + archivePath).initCause(e);
            }
        }
        finally {
            file.close();
            if (!success) {
                this.archiveFileManager.delete(archivePath);
            }
        }
    }

    ArchiveTime getArchiveTime(ArchivePath archivePath) throws FileNotFoundException, IOException {
        return this.archiveFileManager.getTime(archivePath);
    }

    void remove(ArchivePath archivePath) throws FileSystemException, IOException {
        this.archiveFileManager.deleteIfExists(archivePath);
    }

    void watchArchive(Server server) throws IOException, InterruptedException {
        new ArchiveWatcher(server);
    }

    Path getHiddenPath(Path path) {
        return ArchiveFile.hide(this.rootDir, new ArchivePath(path, this.rootDir));
    }

    Path getVisiblePath(Path path) {
        return ArchiveFile.reveal(this.rootDir, new ArchivePath(path, this.rootDir));
    }

    Path getHiddenAbsolutePath(Path path) {
        return this.rootDir.resolve(this.getHiddenPath(path));
    }

    private void walkDirectory(Path root, final FilePieceSpecSetConsumer consumer, final Filter filter) throws IOException, InterruptedException {
        EnumSet<FileVisitOption> opts = EnumSet.of(FileVisitOption.FOLLOW_LINKS);
        final class ArchiveVisitor
        extends SimpleVisitor {
            ArchiveVisitor() {
            }

            @Override
            public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attributes) {
                if (Thread.currentThread().isInterrupted()) {
                    return FileVisitResult.TERMINATE;
                }
                if (ArchiveFile.isHidden(Archive.this.rootDir, dir)) {
                    return FileVisitResult.SKIP_SUBTREE;
                }
                logger.trace("Visiting directory: {}", dir);
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult visitFile(Path path, BasicFileAttributes attributes) {
                if (Thread.currentThread().isInterrupted()) {
                    return FileVisitResult.TERMINATE;
                }
                if (attributes.isRegularFile()) {
                    ArchiveTime archiveTime = new ArchiveTime(attributes);
                    try {
                        archiveTime.setTime(path);
                        ArchivePath archivePath = new ArchivePath(path, Archive.this.rootDir);
                        if (filter.matches(archivePath)) {
                            FileId fileId = new FileId(archivePath, archiveTime);
                            FileInfo fileInfo = archivePath.startsWith(Archive.this.adminDir) ? new FileInfo(fileId, attributes.size(), 131072, -1) : new FileInfo(fileId, attributes.size(), 131072);
                            FilePieceSpecSet specSet = FilePieceSpecSet.newInstance(fileInfo, true);
                            logger.trace("Visiting file: {}", path);
                            consumer.consume(specSet);
                        }
                    }
                    catch (IOException e) {
                        logger.error("Couldn't adjust time of file {}: {}", path, (Object)e.toString());
                    }
                    catch (InterruptedException e) {
                        Thread.currentThread().interrupt();
                        return FileVisitResult.TERMINATE;
                    }
                }
                return FileVisitResult.CONTINUE;
            }
        }
        Files.walkFileTree(root, opts, Integer.MAX_VALUE, new ArchiveVisitor());
        if (Thread.currentThread().isInterrupted()) {
            logger.trace("Interrupted: {}", (Object)this.toString());
            throw new InterruptedException();
        }
    }

    void walkArchive(FilePieceSpecSetConsumer consumer, Filter filter) throws IOException, InterruptedException {
        this.walkDirectory(this.rootDir, consumer, filter);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void addDataProductListener(DataProductListener dataProductListener) {
        List<DataProductListener> list = this.dataProductListeners;
        synchronized (list) {
            this.dataProductListeners.add(dataProductListener);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void removeDataProductListener(DataProductListener dataProductListener) {
        List<DataProductListener> list = this.dataProductListeners;
        synchronized (list) {
            this.dataProductListeners.remove(dataProductListener);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static boolean isEmpty(Path dir) throws IOException {
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(dir);){
            boolean bl = !stream.iterator().hasNext();
            return bl;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void close() throws IOException, InterruptedException {
        try {
            this.delayedPathActionQueue.stop();
        }
        finally {
            this.archiveFileManager.closeAll();
        }
    }

    public String toString() {
        return "Archive [rootDir=" + this.rootDir + "]";
    }

    static {
        Preferences prefs = Preferences.userNodeForPackage(Archive.class);
        ACTIVE_FILE_CACHE_SIZE = prefs.getInt(ACTIVE_FILE_CACHE_SIZE_KEY, 512);
        if (ACTIVE_FILE_CACHE_SIZE <= 0) {
            throw new IllegalArgumentException("Invalid user-preference \"active file cache size\": " + ACTIVE_FILE_CACHE_SIZE);
        }
    }

    @ThreadSafe
    final class ArchiveFileManager {
        @GuardedBy(value="itself")
        private final ArchiveFileMap openSegmentedFiles;

        ArchiveFileManager(int maxNumOpenFiles) {
            if (maxNumOpenFiles <= 0) {
                throw new IllegalArgumentException();
            }
            this.openSegmentedFiles = new ArchiveFileMap(maxNumOpenFiles);
        }

        BulkArchiveFile getForReading(ArchivePath archivePath) throws FileNotFoundException, FileSystemException, IOException {
            BulkArchiveFile file = new BulkArchiveFile(Archive.this.rootDir, archivePath, true);
            return file;
        }

        BulkArchiveFile getForWriting(ArchivePath archivePath) throws FileNotFoundException, FileSystemException, IOException {
            BulkArchiveFile file = new BulkArchiveFile(Archive.this.rootDir, archivePath, false);
            return file;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         * Enabled aggressive block sorting
         * Enabled unnecessary exception pruning
         * Enabled aggressive exception aggregation
         */
        public SegmentedArchiveFile get(FileInfo fileInfo, boolean readonly) throws FileInfoMismatchException, FileSystemException, IOException {
            ArchivePath archivePath = fileInfo.getPath();
            ArchiveFileMap archiveFileMap = this.openSegmentedFiles;
            synchronized (archiveFileMap) {
                while (true) {
                    FileInfo archiveFileInfo;
                    SegmentedArchiveFile file;
                    if ((file = (SegmentedArchiveFile)this.openSegmentedFiles.get(archivePath)) == null) {
                        file = this.getArchiveFile(fileInfo, readonly);
                        if (file == null) {
                            return null;
                        }
                        ArchiveFile prevFile = this.openSegmentedFiles.put(archivePath, file);
                        assert (prevFile == null);
                    }
                    if (fileInfo.equals(archiveFileInfo = file.getFileInfo())) {
                        file.lock();
                        return file;
                    }
                    int cmp = fileInfo.getTime().compareTo(file.getTime());
                    if (readonly || cmp < 0) {
                        return null;
                    }
                    if (cmp <= 0) {
                        throw new FileInfoMismatchException(fileInfo, archiveFileInfo);
                    }
                    try {
                        file.deleteIfExists();
                        continue;
                    }
                    finally {
                        this.openSegmentedFiles.remove(archivePath);
                        continue;
                    }
                    break;
                }
            }
        }

        private SegmentedArchiveFile getArchiveFile(FileInfo fileInfo, boolean readonly) throws FileSystemException, IOException {
            ArchiveFileMap archiveFileMap = this.openSegmentedFiles;
            synchronized (archiveFileMap) {
                while (true) {
                    try {
                        return SegmentedArchiveFile.newInstance(Archive.this.rootDir, fileInfo, readonly);
                    }
                    catch (FileSystemException e) {
                        if (this.removeLru() != null) continue;
                        throw e;
                    }
                    break;
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private SegmentedArchiveFile removeLru() throws IOException {
            ArchiveFileMap archiveFileMap = this.openSegmentedFiles;
            synchronized (archiveFileMap) {
                Iterator iter = this.openSegmentedFiles.entrySet().iterator();
                if (!iter.hasNext()) {
                    return null;
                }
                SegmentedArchiveFile file = (SegmentedArchiveFile)iter.next().getValue();
                file.close();
                iter.remove();
                return file;
            }
        }

        ArchiveTime getTime(ArchivePath archivePath) throws FileSystemException, IOException {
            ArchiveFileMap archiveFileMap = this.openSegmentedFiles;
            synchronized (archiveFileMap) {
                SegmentedArchiveFile file = (SegmentedArchiveFile)this.openSegmentedFiles.get(archivePath);
                if (file != null) {
                    return file.getTime();
                }
                while (true) {
                    try {
                        return BulkArchiveFile.getTime(Archive.this.rootDir, archivePath);
                    }
                    catch (FileSystemException e) {
                        if (this.removeLru() != null) continue;
                        throw e;
                    }
                    break;
                }
            }
        }

        void delete(ArchivePath archivePath) throws IOException {
            while (true) {
                try {
                    BulkArchiveFile.delete(Archive.this.rootDir, archivePath);
                }
                catch (FileSystemException e) {
                    if (this.removeLru() != null) continue;
                    throw e;
                }
                break;
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void deleteIfExists(ArchivePath archivePath) throws FileSystemException, IOException {
            ArchiveFileMap archiveFileMap = this.openSegmentedFiles;
            synchronized (archiveFileMap) {
                SegmentedArchiveFile file = (SegmentedArchiveFile)this.openSegmentedFiles.get(archivePath);
                if (file != null) {
                    file.deleteIfExists();
                    this.openSegmentedFiles.remove(archivePath);
                } else {
                    while (true) {
                        try {
                            SegmentedArchiveFile.deleteIfExists(Archive.this.rootDir, archivePath);
                        }
                        catch (FileSystemException e) {
                            if (this.removeLru() != null) continue;
                            throw e;
                        }
                        break;
                    }
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void closeAll() throws IOException {
            ArchiveFileMap archiveFileMap = this.openSegmentedFiles;
            synchronized (archiveFileMap) {
                Iterator iter = this.openSegmentedFiles.entrySet().iterator();
                while (iter.hasNext()) {
                    ((SegmentedArchiveFile)iter.next().getValue()).close();
                    iter.remove();
                }
            }
        }

        private final class ArchiveFileMap
        extends LinkedHashMap<ArchivePath, SegmentedArchiveFile> {
            private static final long serialVersionUID = 1L;
            private final int maxNumOpenFiles;

            private ArchiveFileMap(int maxNumOpenFiles) {
                super(maxNumOpenFiles, 0.75f, true);
                this.maxNumOpenFiles = maxNumOpenFiles;
            }

            @Override
            protected boolean removeEldestEntry(Map.Entry<ArchivePath, SegmentedArchiveFile> entry) {
                if (this.size() > this.maxNumOpenFiles) {
                    ArchiveFile archiveFile = entry.getValue();
                    try {
                        archiveFile.close();
                    }
                    catch (NoSuchFileException e) {
                        logger.error("File deleted by another thread: \"{}\"", archiveFile);
                    }
                    catch (IOException e) {
                        logger.error("Couldn't close file \"" + archiveFile + "\"", e);
                    }
                    return true;
                }
                return false;
            }
        }
    }

    @ThreadSafe
    private static final class SegmentedArchiveFile
    extends ArchiveFile {
        @GuardedBy(value="lock")
        private FiniteBitSet indexes;
        @GuardedBy(value="lock")
        private FileInfo fileInfo;
        private final ReentrantLock lock = new ReentrantLock();

        private SegmentedArchiveFile(Path rootDir, FileInfo fileInfo) {
            super(rootDir, fileInfo.getPath());
        }

        static SegmentedArchiveFile newInstance(Path rootDir, FileInfo fileInfo, boolean readonly) throws FileSystemException, IOException {
            SegmentedArchiveFile file;
            block13: {
                ArchivePath archivePath = fileInfo.getPath();
                file = new SegmentedArchiveFile(rootDir, fileInfo);
                Path path = SegmentedArchiveFile.hide(rootDir, archivePath);
                if (Files.exists(path, new LinkOption[0])) {
                    try {
                        file.openHiddenFile(fileInfo);
                    }
                    catch (FileNotFoundException e) {
                        logger.debug("Hidden file just deleted by another thread: {}", path);
                        if (!readonly) {
                            logger.debug("Creating new hidden file.");
                            file.createHiddenFile(fileInfo);
                        }
                        break block13;
                    }
                    catch (BadHiddenFileException e) {
                        logger.debug("Deleting corrupt hidden file {}: {}", path, (Object)e.toString());
                        try {
                            Files.delete(path);
                        }
                        catch (IOException e2) {
                            logger.debug("Couldn't delete hidden file {}: {}", path, (Object)e2.toString());
                        }
                        if (!readonly) {
                            logger.debug("Creating new hidden file");
                            file.createHiddenFile(fileInfo);
                        }
                        break block13;
                    }
                }
                path = SegmentedArchiveFile.reveal(rootDir, archivePath);
                if (!Files.exists(path, new LinkOption[0])) {
                    if (!readonly) {
                        file.createHiddenFile(fileInfo);
                    }
                } else {
                    try {
                        file.openVisibleFile(fileInfo);
                    }
                    catch (FileNotFoundException e) {
                        logger.debug("Visible file just deleted by another thread: {}", path);
                        if (readonly) break block13;
                        logger.debug("Creating new hidden file");
                        file.createHiddenFile(fileInfo);
                    }
                }
            }
            return file.randomFile == null ? null : file;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void openVisibleFile(FileInfo template) throws FileSystemException, FileNotFoundException, IOException {
            this.lock();
            try {
                ArchivePath archivePath = template.getPath();
                this.path = SegmentedArchiveFile.reveal(this.rootDir, archivePath);
                this.isVisible = true;
                RandomAccessFile randomFile = new RandomAccessFile(this.path.toFile(), "r");
                FileId fileId = new FileId(archivePath, new ArchiveTime(this.path));
                this.fileInfo = new FileInfo(fileId, randomFile.length(), template.getPieceSize(), template.getTimeToLive());
                this.indexes = new CompleteBitSet(this.fileInfo.getPieceCount());
                this.randomFile = randomFile;
            }
            finally {
                this.unlock();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void createHiddenFile(FileInfo fileInfo) throws FileSystemException, IOException {
            this.lock();
            try {
                this.path = SegmentedArchiveFile.hide(this.rootDir, this.archivePath);
                this.fileInfo = fileInfo;
                this.isVisible = false;
                Files.createDirectories(this.path.getParent(), new FileAttribute[0]);
                this.indexes = new PartialBitSet(fileInfo.getPieceCount());
                this.randomFile = new RandomAccessFile(this.path.toFile(), "rw");
            }
            finally {
                this.unlock();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void openHiddenFile(FileInfo template) throws FileSystemException, FileNotFoundException, BadHiddenFileException, IOException {
            this.lock();
            try {
                this.path = SegmentedArchiveFile.hide(this.rootDir, this.archivePath);
                this.isVisible = false;
                boolean closeIt = true;
                RandomAccessFile randomFile = new RandomAccessFile(this.path.toFile(), "rw");
                try {
                    FileInputStream inputStream;
                    long fileLength;
                    long longSize = 8L;
                    try {
                        fileLength = randomFile.length();
                    }
                    catch (IOException e) {
                        throw (IOException)new IOException("Couldn't get length of file \"" + this.path + "\"").initCause(e);
                    }
                    long pos = fileLength - 8L;
                    if (pos < 0L) {
                        throw new BadHiddenFileException(this.path, "Hidden file too small to be valid");
                    }
                    try {
                        randomFile.seek(pos);
                    }
                    catch (IOException e) {
                        throw (IOException)new IOException("Couldn't seek to byte " + pos + " in file \"" + this.path + "\"").initCause(e);
                    }
                    try {
                        pos = randomFile.readLong();
                    }
                    catch (IOException e) {
                        throw (IOException)new IOException("Couldn't read byte-offset of metadata in file \"" + this.path + "\"").initCause(e);
                    }
                    if (pos < 0L || pos >= fileLength) {
                        throw new BadHiddenFileException(this.path, "Invalid metadata offset: " + pos);
                    }
                    try {
                        randomFile.seek(pos);
                    }
                    catch (IOException e) {
                        throw (IOException)new IOException("Couldn't seek to byte " + pos + " in file \"" + this.path + "\"").initCause(e);
                    }
                    try {
                        inputStream = new FileInputStream(randomFile.getFD());
                    }
                    catch (IOException e) {
                        throw (IOException)new IOException("Couldn't get file-descriptor for file \"" + this.path + "\"").initCause(e);
                    }
                    try {
                        ObjectInputStream ois;
                        try {
                            ois = new ObjectInputStream(inputStream);
                        }
                        catch (StreamCorruptedException e) {
                            throw (BadHiddenFileException)new BadHiddenFileException(this.path, "Couldn't get object-input-stream for file").initCause(e);
                        }
                        catch (IOException e) {
                            throw (IOException)new IOException("Couldn't get object-input-stream for file \"" + this.path + "\"").initCause(e);
                        }
                        try {
                            this.fileInfo = (FileInfo)ois.readObject();
                            this.indexes = (FiniteBitSet)ois.readObject();
                            this.randomFile = randomFile;
                            closeIt = false;
                        }
                        catch (ClassNotFoundException e) {
                            throw (BadHiddenFileException)new BadHiddenFileException(this.path, "Couldn't read file metadata").initCause(e);
                        }
                        catch (ObjectStreamException e) {
                            throw (BadHiddenFileException)new BadHiddenFileException(this.path, "Couldn't read file metadata").initCause(e);
                        }
                        catch (IOException e) {
                            throw (IOException)new IOException("Couldn't read file metadata: " + this.path).initCause(e);
                        }
                        finally {
                            if (closeIt) {
                                try {
                                    ois.close();
                                }
                                catch (IOException ignored) {}
                                closeIt = false;
                            }
                        }
                    }
                    finally {
                        if (closeIt) {
                            try {
                                inputStream.close();
                            }
                            catch (IOException ignored) {}
                            closeIt = false;
                        }
                    }
                }
                finally {
                    if (closeIt) {
                        try {
                            randomFile.close();
                        }
                        catch (IOException ignored) {}
                    }
                }
            }
            finally {
                this.unlock();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        ArchiveTime getTime() {
            this.lock();
            try {
                ArchiveTime archiveTime = this.fileInfo.getTime();
                return archiveTime;
            }
            finally {
                this.unlock();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        FileInfo getFileInfo() {
            this.lock();
            try {
                FileInfo fileInfo = this.fileInfo;
                return fileInfo;
            }
            finally {
                this.unlock();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        boolean willMakeComplete(PieceSpec pieceSpec) {
            this.lock();
            try {
                int index = pieceSpec.getIndex();
                boolean bl = !this.indexes.isSet(index) && this.indexes.getSetCount() == this.indexes.getSize() - 1;
                return bl;
            }
            finally {
                this.unlock();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        boolean putPiece(Piece piece) throws FileSystemException, IOException {
            this.lock();
            try {
                int index = piece.getIndex();
                if (!this.indexes.isSet(index)) {
                    this.randomFile.seek(piece.getOffset());
                    this.randomFile.write(piece.getData());
                    this.indexes = this.indexes.setBit(index);
                    if (this.indexes.areAllSet()) {
                        this.close();
                        assert (this.isVisible);
                        this.openVisibleFile(this.fileInfo);
                    }
                }
                boolean bl = this.isVisible;
                return bl;
            }
            finally {
                this.unlock();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        boolean hasPiece(int index) {
            this.lock();
            try {
                boolean bl = this.indexes.isSet(index);
                return bl;
            }
            finally {
                this.unlock();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        Piece getPiece(PieceSpec pieceSpec) throws FileSystemException, IOException {
            this.lock();
            try {
                assert (this.indexes.isSet(pieceSpec.getIndex()));
                byte[] data = new byte[pieceSpec.getSize()];
                this.randomFile.seek(pieceSpec.getOffset());
                int nbytes = this.randomFile.read(data);
                assert (nbytes == data.length);
                Piece piece = new Piece(pieceSpec, data);
                return piece;
            }
            finally {
                this.unlock();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        protected void close() throws IOException {
            this.lock();
            try {
                if (this.randomFile != null) {
                    if (this.isVisible) {
                        this.randomFile.close();
                    } else {
                        long length = this.fileInfo.getSize();
                        if (!this.indexes.areAllSet()) {
                            this.randomFile.seek(length);
                            FileOutputStream outputStream = new FileOutputStream(this.randomFile.getFD());
                            ObjectOutputStream oos = new ObjectOutputStream(outputStream);
                            oos.writeObject(this.fileInfo);
                            oos.writeObject(this.indexes);
                            oos.writeLong(length);
                            oos.close();
                            this.fileInfo.getTime().setTime(this.path);
                        } else {
                            this.randomFile.getChannel().truncate(length);
                            this.randomFile.close();
                            Path newPath = SegmentedArchiveFile.reveal(this.rootDir, this.archivePath);
                            block7: while (true) {
                                try {
                                    while (true) {
                                        Files.createDirectories(newPath.getParent(), new FileAttribute[0]);
                                        try {
                                            this.fileInfo.getTime().setTime(this.path);
                                            Files.move(this.path, newPath, StandardCopyOption.ATOMIC_MOVE, StandardCopyOption.REPLACE_EXISTING);
                                            this.path = newPath;
                                            this.isVisible = true;
                                            logger.debug("Complete file: {}", this.fileInfo);
                                            break block7;
                                        }
                                        catch (NoSuchFileException e) {
                                            logger.trace("Directory in path just deleted by another thread: {}", newPath);
                                            continue;
                                        }
                                        break;
                                    }
                                }
                                catch (NoSuchFileException e) {
                                    logger.trace("Directory in path just deleted by another thread: {}", newPath);
                                    continue;
                                }
                                break;
                            }
                        }
                    }
                    this.randomFile = null;
                }
            }
            finally {
                this.unlock();
            }
        }

        void lock() {
            this.lock.lock();
        }

        void unlock() {
            this.lock.unlock();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        protected void deleteIfExists() throws IOException {
            this.lock();
            try {
                try {
                    this.close();
                }
                catch (IOException iOException) {
                    // empty catch block
                }
                Files.deleteIfExists(this.path);
            }
            finally {
                this.unlock();
            }
        }

        static void deleteIfExists(Path rootDir, ArchivePath archivePath) throws IOException {
            Path path = archivePath.getAbsolutePath(rootDir);
            try {
                Files.deleteIfExists(path);
            }
            catch (NoSuchFileException e) {
                logger.trace("File doesn't exist: {}", archivePath);
            }
        }
    }

    private static class BadHiddenFileException
    extends IOException {
        private static final long serialVersionUID = 1L;
        private final Path path;

        BadHiddenFileException(Path path, String msg) {
            super(msg);
            this.path = path;
        }

        @Override
        public String toString() {
            return this.getMessage() + " [path=" + this.path + "]";
        }
    }

    @ThreadSafe
    private static final class BulkArchiveFile
    extends ArchiveFile {
        BulkArchiveFile(Path rootDir, ArchivePath archivePath, boolean readonly) throws FileNotFoundException, FileSystemException, IOException {
            super(rootDir, archivePath);
            if (readonly) {
                this.initFromVisibleFile();
            } else {
                this.initByCreatingHiddenFile();
            }
        }

        private synchronized void initFromVisibleFile() throws FileSystemException, FileNotFoundException, IOException {
            this.path = BulkArchiveFile.reveal(this.rootDir, this.archivePath);
            this.isVisible = true;
            this.randomFile = new RandomAccessFile(this.path.toFile(), "r");
        }

        private synchronized void initByCreatingHiddenFile() throws FileSystemException, IOException {
            this.path = BulkArchiveFile.hide(this.rootDir, this.archivePath);
            Files.createDirectories(this.path.getParent(), new FileAttribute[0]);
            this.isVisible = false;
            this.randomFile = new RandomAccessFile(this.path.toFile(), "rw");
        }

        static ArchiveTime getTime(Path rootDir, ArchivePath archivePath) throws FileSystemException, NoSuchFileException, IOException {
            return new ArchiveTime(archivePath.getAbsolutePath(rootDir));
        }

        synchronized void transferFrom(ReadableByteChannel channel) throws IOException {
            this.randomFile.getChannel().transferFrom(channel, 0L, Long.MAX_VALUE);
        }

        synchronized InputStream asInputStream() throws IOException {
            if (!this.isVisible) {
                throw new IllegalStateException();
            }
            this.randomFile.seek(0L);
            return Channels.newInputStream(this.randomFile.getChannel());
        }

        @Override
        protected synchronized void close() throws IOException {
            if (this.randomFile != null) {
                this.randomFile.close();
                this.randomFile = null;
                if (!this.isVisible) {
                    Path newPath = BulkArchiveFile.reveal(this.rootDir, this.archivePath);
                    block4: while (true) {
                        try {
                            while (true) {
                                Files.createDirectories(newPath.getParent(), new FileAttribute[0]);
                                try {
                                    logger.debug("Revealing file: {}", newPath);
                                    Files.move(this.path, newPath, StandardCopyOption.ATOMIC_MOVE, StandardCopyOption.REPLACE_EXISTING);
                                    this.path = newPath;
                                    this.isVisible = true;
                                    break block4;
                                }
                                catch (NoSuchFileException e) {
                                    logger.trace("Directory in path just deleted by another thread: {}", newPath);
                                    continue;
                                }
                                break;
                            }
                        }
                        catch (NoSuchFileException e) {
                            logger.trace("Directory in path just deleted by another thread: {}", newPath);
                            continue;
                        }
                        break;
                    }
                }
            }
        }

        @Override
        protected synchronized void deleteIfExists() throws IOException {
            try {
                this.close();
            }
            catch (IOException iOException) {
                // empty catch block
            }
            Files.deleteIfExists(this.path);
        }

        static void delete(Path rootDir, ArchivePath archivePath) throws IOException {
            Path path = archivePath.getAbsolutePath(rootDir);
            if (!(Files.deleteIfExists(path) || Files.deleteIfExists(path = BulkArchiveFile.hide(rootDir, archivePath)) || Files.deleteIfExists(path = BulkArchiveFile.reveal(rootDir, archivePath)))) {
                logger.debug("File doesn't exist: {}", path);
            }
            Path dir = null;
            try {
                for (dir = path.getParent(); dir != null && !Files.isSameFile(dir, rootDir) && Archive.isEmpty(dir); dir = dir.getParent()) {
                    try {
                        if (Files.deleteIfExists(dir)) continue;
                        logger.debug("Directory was just deleted by another thread: {}", dir);
                        break;
                    }
                    catch (DirectoryNotEmptyException ignored) {
                        logger.debug("Not deleting directory because it was just added-to by another thread: {}", dir);
                    }
                }
            }
            catch (NoSuchFileException ignored) {
                logger.debug("Directory was just deleted by another thread: {}", dir);
            }
        }
    }

    @ThreadSafe
    static abstract class ArchiveFile {
        @GuardedBy(value="lock")
        protected Path path;
        @GuardedBy(value="lock")
        protected RandomAccessFile randomFile;
        @GuardedBy(value="lock")
        protected boolean isVisible;
        protected final ArchivePath archivePath;
        protected final Path rootDir;

        ArchiveFile(Path rootDir, ArchivePath archivePath) {
            if (!rootDir.isAbsolute()) {
                throw new IllegalArgumentException();
            }
            this.rootDir = rootDir;
            if (archivePath == null) {
                throw new NullPointerException();
            }
            this.archivePath = archivePath;
        }

        protected abstract void close() throws IOException;

        protected abstract void deleteIfExists() throws IOException;

        protected static Path hide(Path rootDir, ArchivePath archivePath) {
            Path path = archivePath.getPath();
            path = HIDDEN_DIR.resolve(path);
            return rootDir.resolve(path);
        }

        protected static Path reveal(Path rootDir, ArchivePath archivePath) {
            Path path = archivePath.getPath();
            return rootDir.resolve(path);
        }

        static boolean isHidden(Path rootDir, Path path) {
            if (path.isAbsolute()) {
                path = rootDir.relativize(path);
            }
            return null == path ? false : path.startsWith(HIDDEN_DIR);
        }
    }

    @NotThreadSafe
    private final class ArchiveWatcher {
        private final WatchService watchService;
        private final Map<WatchKey, Path> dirs = new HashMap<WatchKey, Path>();
        private final Map<Path, WatchKey> keys = new HashMap<Path, WatchKey>();
        private final Server server;

        ArchiveWatcher(Server server) throws IOException, InterruptedException {
            this.server = server;
            if (null == server) {
                throw new NullPointerException();
            }
            this.watchService = Archive.this.rootDir.getFileSystem().newWatchService();
            try {
                this.registerDirectoryTree(Archive.this.rootDir);
                while (true) {
                    Path dir;
                    WatchKey key = this.watchService.take();
                    for (WatchEvent<?> event : key.pollEvents()) {
                        WatchEvent.Kind<?> kind = event.kind();
                        if (kind == StandardWatchEventKinds.OVERFLOW) {
                            logger.error("Couldn't keep-up watching file-tree rooted at \"{}\"", Archive.this.rootDir);
                            continue;
                        }
                        Path name = (Path)event.context();
                        Path path = this.dirs.get(key);
                        path = path.resolve(name);
                        if (kind == StandardWatchEventKinds.ENTRY_CREATE) {
                            try {
                                this.newFile(path);
                            }
                            catch (IOException e) {
                                logger.error("Error with new file " + path, e);
                            }
                            continue;
                        }
                        if (kind != StandardWatchEventKinds.ENTRY_DELETE) continue;
                        try {
                            this.removedFile(path);
                        }
                        catch (IOException e) {
                            logger.error("Error with removed file \"" + path + "\"", e);
                        }
                    }
                    if (key.reset() || (dir = this.dirs.remove(key)) == null) continue;
                    this.keys.remove(dir);
                }
            }
            catch (Throwable throwable) {
                this.watchService.close();
                throw throwable;
            }
        }

        private void newFile(final Path path) throws IOException {
            Files.walkFileTree(path, new SimpleVisitor(){

                @Override
                public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attributes) {
                    if (ArchiveFile.isHidden(Archive.this.rootDir, dir)) {
                        return FileVisitResult.SKIP_SUBTREE;
                    }
                    try {
                        logger.debug("New directory: {}", path);
                        ArchiveWatcher.this.register(dir);
                    }
                    catch (IOException e) {
                        throw new IOError(e);
                    }
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult visitFile(Path path2, BasicFileAttributes attributes) {
                    assert (attributes.isRegularFile());
                    logger.debug("New file: {}", path2);
                    ArchiveWatcher.this.notifyServerAbout(path2, attributes);
                    return FileVisitResult.CONTINUE;
                }
            });
        }

        private void notifyServerAbout(Path path, BasicFileAttributes attributes) {
            try {
                ArchiveTime.adjustTime(path);
                ArchivePath archivePath = new ArchivePath(path, Archive.this.rootDir);
                FileId fileId = new FileId(archivePath, new ArchiveTime(attributes));
                FileInfo fileInfo = archivePath.startsWith(Archive.this.adminDir) ? new FileInfo(fileId, attributes.size(), 131072, -1) : new FileInfo(fileId, attributes.size(), 131072);
                logger.trace("New file: {}", path);
                this.server.newData(FilePieceSpecSet.newInstance(fileInfo, true));
            }
            catch (IOException e) {
                logger.error("Couldn't set time of file {}: {}", path, (Object)e.toString());
            }
        }

        private void removedFile(Path path) throws IOException {
            ArchivePath archivePath = new ArchivePath(path, Archive.this.rootDir);
            if (!archivePath.startsWith(Archive.this.adminDir)) {
                WatchKey k = this.keys.remove(path);
                if (null != k) {
                    this.dirs.remove(k);
                    k.cancel();
                }
                logger.trace("Removed file: {}", archivePath);
                this.server.removed(archivePath);
            }
        }

        private void register(Path dir) throws IOException {
            if (!this.keys.containsKey(dir)) {
                logger.trace("New directory: {}", dir);
                WatchKey key = dir.register(this.watchService, StandardWatchEventKinds.ENTRY_CREATE, StandardWatchEventKinds.ENTRY_DELETE, StandardWatchEventKinds.OVERFLOW);
                this.dirs.put(key, dir);
                this.keys.put(dir, key);
            }
        }

        private void registerDirectoryTree(Path dir) throws IOException {
            EnumSet<FileVisitOption> opts = EnumSet.of(FileVisitOption.FOLLOW_LINKS);
            Files.walkFileTree(dir, opts, Integer.MAX_VALUE, new SimpleVisitor(){

                @Override
                public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attributes) {
                    if (ArchiveFile.isHidden(Archive.this.rootDir, dir)) {
                        return FileVisitResult.SKIP_SUBTREE;
                    }
                    try {
                        ArchiveWatcher.this.register(dir);
                    }
                    catch (IOException e) {
                        throw new IOError(e);
                    }
                    return FileVisitResult.CONTINUE;
                }
            });
        }

        public String toString() {
            return "ArchiveWatcher [rootDir=" + Archive.this.rootDir + "]";
        }
    }

    @ThreadSafe
    private final class DistributedTrackerFilesFactory {
        private final ConcurrentMap<InetSocketAddress, DistributedTrackerFiles> instances = new ConcurrentHashMap<InetSocketAddress, DistributedTrackerFiles>();

        private DistributedTrackerFilesFactory() {
        }

        DistributedTrackerFiles getInstance(InetSocketAddress trackerAddress) {
            DistributedTrackerFiles prevInstance;
            DistributedTrackerFiles instance = (DistributedTrackerFiles)this.instances.get(trackerAddress);
            if (instance == null && (prevInstance = this.instances.putIfAbsent(trackerAddress, instance = new DistributedTrackerFiles(Archive.this, trackerAddress))) != null) {
                instance = prevInstance;
            }
            return instance;
        }
    }

    @ThreadSafe
    static final class DistributedTrackerFiles {
        private final Archive archive;
        private final DistributedFile<Topology> topologyFile;
        private final DistributedFile<InetSocketAddress> reportingAddressFile;
        private final ObjectLock<Topology> topologyLock = new ObjectLock();
        @GuardedBy(value="this")
        private final Thread distributor = new Distributor();
        @GuardedBy(value="this")
        private final ArchiveTime networkUpdateTime = ArchiveTime.BEGINNING_OF_TIME;
        private final ArchivePath trackerPath;

        DistributedTrackerFiles(Archive archive, InetSocketAddress trackerAddress) {
            this.archive = archive;
            String packagePath = this.getClass().getPackage().getName();
            String packageName = packagePath.substring(packagePath.lastIndexOf(46) + 1);
            packageName = packageName.toUpperCase();
            this.trackerPath = archive.getAdminDir().resolve(new ArchivePath(Paths.get(trackerAddress.getHostString() + "-" + trackerAddress.getPort(), new String[0])));
            ArchivePath topologyArchivePath = this.trackerPath.resolve("topology");
            this.topologyFile = new DistributedFile<Topology>(topologyArchivePath, Topology.class);
            ArchivePath reportingAddressArchivePath = this.trackerPath.resolve("reportingAddress");
            this.reportingAddressFile = new DistributedFile<InetSocketAddress>(reportingAddressArchivePath, InetSocketAddress.class);
        }

        ArchivePath getTopologyArchivePath() {
            return this.topologyFile.getArchivePath();
        }

        ArchiveTime getTopologyArchiveTime() throws IOException {
            return this.topologyFile.getArchiveTime();
        }

        Topology getTopology() throws NoSuchFileException, IOException {
            return this.topologyFile.get();
        }

        InetSocketAddress getReportingAddress() throws NoSuchFileException, IOException {
            return this.reportingAddressFile.get();
        }

        void distribute(Topology topology) {
            if (topology == null) {
                throw new NullPointerException();
            }
            this.ensureDistributorStarted();
            this.topologyLock.put(topology);
        }

        private synchronized void ensureDistributorStarted() {
            if (!this.distributor.isAlive()) {
                this.distributor.setDaemon(true);
                this.distributor.start();
            }
        }

        void distribute(InetSocketAddress reportingAddress) throws FileSystemException, IOException {
            this.reportingAddressFile.set(reportingAddress);
        }

        Filter getFilter() {
            return Filter.getInstance(this.trackerPath.toString());
        }

        public String toString() {
            return "DistributedTrackerFiles [topologyLock=" + this.topologyLock + ", networkUpdateTime=" + this.networkUpdateTime + "]";
        }

        private class DistributedFile<T extends Serializable> {
            private final ArchivePath path;
            private final Class<T> type;
            private ArchiveTime archiveTime = ArchiveTime.BEGINNING_OF_TIME;
            private T object;

            DistributedFile(ArchivePath path, Class<T> type) {
                if (path == null) {
                    throw new NullPointerException();
                }
                this.path = path;
                if (type == null) {
                    throw new NullPointerException();
                }
                this.type = type;
            }

            ArchivePath getArchivePath() {
                return this.path;
            }

            synchronized ArchiveTime getArchiveTime() {
                return this.archiveTime;
            }

            synchronized T get() throws NoSuchFileException, IOException {
                ArchiveTime updateTime = DistributedTrackerFiles.this.archive.getArchiveTime(this.path);
                if (this.archiveTime.compareTo(updateTime) < 0) {
                    try {
                        this.object = (Serializable)this.type.cast(DistributedTrackerFiles.this.archive.restore(this.path, this.type));
                        this.archiveTime = updateTime;
                    }
                    catch (ClassNotFoundException e) {
                        throw (ClassCastException)new ClassCastException("File-object has unknown type").initCause(e);
                    }
                    catch (ClassCastException e) {
                        throw (ClassCastException)new ClassCastException("File-object isn't type " + this.type.getSimpleName()).initCause(e);
                    }
                    catch (FileNotFoundException e) {
                        throw new NoSuchFileException(e.getLocalizedMessage());
                    }
                }
                return this.object;
            }

            synchronized boolean set(T object) throws FileSystemException, IOException {
                ArchiveTime now = new ArchiveTime();
                if (this.archiveTime.compareTo(now) < 0) {
                    DistributedTrackerFiles.this.archive.save(this.path, object);
                    this.archiveTime = DistributedTrackerFiles.this.archive.getArchiveTime(this.path);
                    this.object = object;
                    return true;
                }
                return false;
            }
        }

        private class Distributor
        extends Thread {
            private Distributor() {
            }

            @Override
            public void run() {
                block5: while (true) {
                    try {
                        while (true) {
                            Topology topology = (Topology)DistributedTrackerFiles.this.topologyLock.take();
                            try {
                                if (DistributedTrackerFiles.this.topologyFile.set(topology)) continue block5;
                                logger.debug("Topology-file not distributed because it's not sufficiently new: {}", DistributedTrackerFiles.this.topologyFile.getArchivePath());
                                continue block5;
                            }
                            catch (IOException e) {
                                logger.error("Couldn't save network topology", e);
                                continue;
                            }
                            break;
                        }
                    }
                    catch (InterruptedException e) {
                        logger.error("This thread should not have been interrupted", e);
                        continue;
                    }
                    catch (Throwable t) {
                        logger.error("Unexpected error", t);
                        continue;
                    }
                    break;
                }
            }
        }
    }

    private static abstract class SimpleVisitor
    extends SimpleFileVisitor<Path> {
        private SimpleVisitor() {
        }

        @Override
        public final FileVisitResult visitFileFailed(Path path, IOException e) throws IOException {
            if (e instanceof NoSuchFileException) {
                logger.debug("File was just deleted by another thread: {}", path);
                return FileVisitResult.CONTINUE;
            }
            throw (IOException)new IOException("Couldn't visit file \"" + path + "\"").initCause(e);
        }
    }
}

